% MATLAB software that computes approximation errors in a neoclassical
% stochastic growth model using an alternative method, as described in 
% the article "Lower Bounds on Approximation Errors to Numerical Solutions  
% of Dynamic Economic Models" by Kenneth L. Judd, Lilia Maliar and Serguei 
% Maliar (2016), Econometrica, forthcoming,(henceforth, JMM, 2016). 
%
%
% This version: February 12, 2017. First version: January 20, 2012.
% 
% -------------------------------------------------------------------------
% The software uses the following files: 
% -------------------------------------------------------------------------
% 1. "Main_Neocl_Expect.m"       is a main file for computing approximation
%                                errors in the variables, as is described in 
%                                Appendix A3; and residuals in the optimality 
%                                conditions for the neoclassical stochastic  
%                                growth model; it uses a Dynare routine 
%                                "Dynare_Neocl.mod" to find a perturbation 
%                                solution; it uses stochastic simulation 
%                                points to test the accuracy
% 2. "Dynare_Neocl.mod"          is a Dynare routine that produces first-
%                                and second-order perturbation solutions
% 3. "Simulation_Neocl.m"        simulates a time series solution to the  
%                                neoclassical stochastic growth model for a  
%                                given sequence of exogenous shocks
% 4. "SSE_Expect.m"              computes the sum of squared approximation 
%                                errors
% 5. "GH_Quadrature.m"           constructs integration nodes and weights for  
%                                the Gauss-Hermite rules with the number of  
%                                nodes in each dimension ranging from one to  
%                                ten; borrowed from the paper "Numerically 
%                                Stable and Accurate Stochastic Simulation 
%                                Approaches for Solving Dynamic Economic 
%                                Models" by Kenneth L. Judd, Lilia Maliar 
%                                and Serguei Maliar, (2011), Quantitative 
%                                Economics 2/2, 173210, henceforth, JMM(2011).    
% 6. "epsi_test.mat"             contains the series of the exogenous shocks 
%                                of length 10,200 for the accuracy test 
% -------------------------------------------------------------------------
% Copyright  2017 by Lilia Maliar and Serguei Maliar. All rights reserved. 
% The code may be used, modified and redistributed under the terms provided 
% in the file "License_Agreement.txt".
% -------------------------------------------------------------------------

clear; close all; clc;


% 1. Parameter values 
% -------------------
gam = 0.1;            % Utility-function parameter
beta  = 0.99;         % Discount factor
alpha = 0.33;         % Share of capital in production
d = 0.025;            % Depreciation rate
rho   = 0.95;         % Autocorrelation coefficient in the process for shocks
sigma = 0.01;         % Volatility of productivity shocks

% 2. Normalizing constant in production
%--------------------------------------
At = (1/beta-1+d)/alpha; 
                      % Level of technology; this normalizing constant is 
                      % allows us to make the steady state level of capital 
                      % be equal to 1
                      
                      
% 3. Saving the parameters into "parametrfile" that will be used by 
% "Dynare_Neocl.mod"
%------------------------------------------------------------------     
save parameterfile gam beta alpha d rho sigma At                 


% 4. Computing a Dynare solution using "Dynare_Neocl.mod"
% -------------------------------------------------------
path(path,'4.3.3\matlab')  
                           % Add a Dynare path
dynare Dynare_Neocl.mod;   % Compute the first- and second-order perturbation 
                           % solutions

% 5. Recovering the matrices that are components of the decision rules 
% produced by Dynare 
%---------------------------------------------------------------------
Order = oo_.dr.order_var;  % Check that the order in which the variables 
                           % appear in decision rules (DR-order) coincides 
                           % with the order in which the variables are declared
                           
% 5.1 Recovering the matrices relevant for the first-order perturbation 
% solution (PER1)
%----------------------------------------------------------------------
% The first-order approximation has the form y(t)=SS+A*(y(t-1)-SS)+B*u(t),
% where y(t) and u(t) are endogenous and exogenous (i.e., shocks) variables,
% resp.                           
SS    = oo_.dr.ys;         % The steady state values of all variables appearing
                           % in the order in which the endogenous variables 
                           % are declared in the Dynare program (there are 
                           % 3 variables, k, a and c); 3-by-1
A     = oo_.dr.ghx;        % A matrix of coefficients whose rows correspond
                           % to all the endogenous variables in the DR-order,
                           % and whose columns correspond to the state
                           % variables in the DR-order; 3-by-2
B     = oo_.dr.ghu;        % A matrix of coefficients whose rows correspond
                           % to all the endogenous variables in the DR-order,
                           % and whose columns correspond to the exogenous
                           % variables (shocks) in the DR-order; 3-by-1

% 5.2 Recovering the matrices relevant for the second-order perturbation 
% solution (PER2)
%-----------------------------------------------------------------------                          
% The second-order approximation has the form: 
% y(t)=SS+0.5*del2+A(y(t-1)-SS)+B*u(t)+0.5*C*(y(t-1)-SS)*kron*(y(t-1)-SS)+0.5*D*
% *(u(t)*kron*u(t)+E*(y(t-1)-SS)*kron*u(t))

del2  = oo_.dr.ghs2;       % A column vector whose rows correspond to all 
                           % endogenous variables in the DR-order; 3-by-1
C     = oo_.dr.ghxx;       % A matrix of coefficients whose row correspond
                           % to all the endogenous variables in the
                           % DR-order, and whose columns correspond to the
                           % Kronecker product of the vector of state
                           % variables (there are 6 of them) in the DR-order; 
                           % 3-by-4
D     = oo_.dr.ghuu;       % A matrix of coefficients whose row correspond
                           % to all the endogenous variables in the
                           % DR-order, and whose columns correspond to the
                           % Kronecker product of the exogenous variables
                           % (there is 1 of them) in the declaration order; 
                           % 3-by-1
E     = oo_.dr.ghxu;       % A matrix of coefficients whose row correspond
                           % to all the endogenous variables in the
                           % DR-order, and whose columns correspond to the
                           % Kronecker product of the vector of state
                           % variables (there are 2 of them) in the DR-order 
                           % by the vector of exogenous variables (there is
                           % 1 of them) in the declaration order; 3-by-2

% 6. Simulating the time-series solution for the accuracy tests
%--------------------------------------------------------------
T_test = 10200;            % Length of simulations in the test
%epsi_test  = rand(T_test,1);   
                           % A draw of uniformly distributed random numbers of 
                           % length T_test for evaluating the accuracy

load epsi_test;            % Load previously stored sequences of uniformly 
                           % distributed random numbers "epsi_test" for 
                           % evaluating the accuracy

[PER1_test, PER2_test]  = Simulation_Neocl(SS,del2,A,B,C,D,E,epsi_test,sigma,T_test);
                           % We simulate the Dynare solution "manually" 
                           % instead of using a Dynare simulation routine
                           
% 7. Gauss-Hermite quadrature 
% ---------------------------
Qn = 10;                   % Number of nodes in each dimension; this Gauss-
                           % Hermite quadrature method is used both to
                           % evaluate the integrals when computing
                           % residuals and to find approximation errors in 
                           % future variables
vcv = sigma^2;             % Variance of the productivity shock
[n_nodes,epsi_norm,weights_norm] = GH_Quadrature(Qn,1,vcv);
                           % Gauss Hermite quadrature: number of nodes, 
                           % values of nodes and their weights, respectively;
                           
% 8. Creating an optimization option structure for function quadprog
%-------------------------------------------------------------------
options = optimset('Algorithm','interior-point-convex','LargeScale', 'on', 'Display', 'off'); 
                           % Set the optimization options

% _________________________________________________________________
%
% Approximation errors and residuals of the first- and second-order
% perturbation solutions
% __________________________________________________________________                           
order = 2;                 % The maximum order of perturbation considered

% 9. Initializing the matrices for the results on approximation errors and 
% residuals
%-------------------------------------------------------------------------
Q_Mean_all = zeros(order,3);  
Q_Mean_1 = zeros(order,1); 
Q_Max_all = zeros(order,3);
Q_Max_1 = zeros(order,1);
N_Mean_all = zeros(order,3);
N_Mean_1 = zeros(order,1); 
N_Max_all = zeros(order,3);
N_Max_1 = zeros(order,1);
Mean_2 = zeros(order,2);
Mean_1 = zeros(order,1); 
Max_2 = zeros(order,2);
Max_1 = zeros(order,1);
time_PER = zeros(order,1);

% 10. Choosing the appropriate perturbation solution 
%---------------------------------------------------
for i = 1:order            % For a Dynare solution of orders "order"; the 
                           % maximum value of "order" is 2,...
    
    if i == 1              % If the considered perturbation solution is of 
                           % the first order, ...
       PER = PER1_test;    % Use the corresponding simulated time series
    elseif i == 2          % If the considered perturbation solution is of 
                           % the second order, ...
       PER = PER2_test;    % Use the corresponding simulated time series
    end

% 11. Compute approximation errors, deltas, and residuals for each data point
%----------------------------------------------------------------------------
tic;
for t = 1:T_test;                 % For each period, ... 

     % 11.1. Current variables known at t (theta0, c0, k1) or future variables  
     % that directly follow from them (theta1)
     %------------------------------------------------------------------
       
       theta0 = PER(2,t);  % Current-period productivity level theta(t) is  
                           % the second variable in PER
       theta1(1:n_nodes,1) = theta0*rho + epsi_norm; 
                           % Compute the next-period productivity level
                           % theta(t+1) in all integration nodes using the 
                           % process (21) in JMM (2016)
       c0 = PER(3,t);      % Current consumption is the third variable in PER
       k1 = PER(1,t);      % Next-period capital is the first variable in PER
       k1_dupl = k1*ones(n_nodes,1); 
                           % Duplicate next-period capital; n_nodes-by-1
                           
     % 11.2 Consumption chosen at t+1
     % ------------------------------  
        
       % Simulating the first-order perturbation solution
       %-------------------------------------------------
       if i == 1
          DEV_all = PER(:,t) - SS;
                           % Compute a deviation of the vector of all 
                           % variables from the steady state, i.e., y(t)-SS
          DEV = DEV_all(1:2); 
                           % Consider y(t)-SS for the state variables
          for j = 1:n_nodes;
              EPS = epsi_norm(j,1);
                           % Form a matrix of exogenous variables (shocks) 
              PER_nodes(:,j) = SS + A*DEV + B*EPS;
                           % Compute PER1 (see a comment in Section 5.1 above)
          end
          
        % Simulating the second-order perturbation solution in integration 
        % nodes
        %-----------------------------------------------------------------
        elseif i == 2  % Simulate the second-order perturbation solution
          DEV_all = PER(:,t)-SS;
                           % Compute a deviation of the vector of all 
                           % variables from the steady state, i.e., y(t)-SS
          DEV = DEV_all(1:2); 
                           % Consider y(t)-SS for the state variables
          for j = 1:n_nodes 
              EPS = epsi_norm(j,1);
                           % Form a matrix of exogenous variables (shocks) 
              DEV2 = kron(DEV,DEV);
                           % Compute a Kronecker product of the vector 
                           % of state variables 
              EPS2 = kron(EPS,EPS);
                           % Compute a Kronecker product of the exogenous 
                           % variables
              DEVEPS = kron(DEV,EPS);
                           % Compute a Kronecker product of the vector of state
                           % variables by the vector of exogenous variables 
              PER_nodes(:,j) = SS + A*DEV + B*EPS + 0.5*del2 + 0.5*C*DEV2 + 0.5*D*EPS2 + E*DEVEPS;
                           % Compute PER2 (see a comment in Section 5.2 above)
          end
        end

        % Computing next-period consumption in the integration nodes
        %-----------------------------------------------------------
        c1 = PER_nodes(3,:)';
                           % Next-period consumption in each 
                           % integration node; n_nodes-by-1
                                   
      % 11.3 Computing approximation errors and residuals for all t>=2
      %---------------------------------------------------------------
                                   
        if t>=2;           % t=1 is an initial period; thus, we start from t=2
                                      
        k0 = PER(1,t-1);   % Beginning-of-the-period capital; capital is the 
                           % first variable in PER
  
      
       % Computing the coefficients for the minimization problem (28) in 
       % JMM (2016)
       %----------------------------------------------------------------
       h1(t-1,1) = (beta*weights_norm'*(c1.^(-gam)./c0.^(-gam).*(1-d+alpha*At.*exp(theta1).*k1_dupl.^(alpha-1))));
                           % Conditional expectation h1 defined in Appendix  
                           % A1 in JMM (2016)
                                      
        a11 = gam;         % Coefficient        
        a13 = 1;           % Coefficient 
        b1 = log(h1(t-1,1));
                           % Coefficient 
     
       a21 = c0;                   % Coefficient a21 in eq. (A4) in JMM (2016)
       a22 = k1;                   % Coefficient a22 in eq. (A4) in JMM (2016)
       b2 = (1-d)*k0+exp(theta0)*At*k0^alpha-c0-k1;
                                   % Coefficient b2 in eq. (A4) in JMM (2016)
        
        
      % Computing the residuals in the FOSs
      %------------------------------------
      ResidualsEeq(t-1,1) = (beta*weights_norm'*(c1.^(-gam).*(1-d+alpha*At.*exp(theta1).*k1_dupl.^(alpha-1)))).^(-1/gam)./c0-1;
                                   % Residual in the Euler equation (22) in  
                                   % every period of time; see eq. (31) in 
                                   % JMM (2016)
      ResidualsBC(t-1,1) = (At*exp(theta0)*k0^alpha+k0*(1-d)-c0)/k1-1; 
                                   % Residuals in the budget constraint (20) 
                                   % in every period of time; see eq. (32) 
                                   % in JMM (2016)
                                                         
 
      % Quadratic programming: minimizing the sum of squared deltas s.t. 2 
      % linear constraints for deltak1 and deltac0, obtained from (A2) and 
      % (A4)
      %--------------------------------------------------------------------
      % Solve the quadratic minimization problem min 0.5*x'*H*x + f'*x 
      % subject to:  A*x <= b; our problem is given by (A5)
        H = eye(3,3);
        f = [0;0;0]; 
        Aeq = [a11 0 a13; a21 a22 0];
        beq = [b1; b2];
        warning('off','all');  
        [x,fval,exitflag,output,lambda] = quadprog(H,f,[],[],Aeq,beq,-1e-3,1e+3,[],options);
                                   % Solve a quadratic minimization                         
        Q_deltac0(t-1,1) = x(1,1); % Approximation error in c(t)
        Q_deltak1(t-1,1) = x(2,1); % Approximation error in k(t+1)
        Q_deltaE0(t-1,1) = x(3,1); % Approximation error in E(t)
                                   
     
        % Non-linear solution: minimize the sum of squared deltas s.t. 2 
        % non-linear constraints for deltak1 and deltaE0
        %---------------------------------------------------------------
        exp1(t-1,1) = (beta*weights_norm'*(c1.^(-gam).*(1-d)));
                                   % Conditional expectation term in (A9)
        exp2(t-1,1) =(beta*weights_norm'*(c1.^(-gam).*exp(theta1)*alpha*At*k1^(alpha-1)));
                                   % Conditional expectation term in (A9)
        z = fminsearch(@(z) SSE_Expect(z,exp1(t-1,1),exp2(t-1,1),k0,c0,theta0,k1,d,alpha,At,gam), Q_deltac0(t-1,1));
                                   % Compute deltas using fminsearch
        N_deltac0(t-1,1) = z;     % Approximation error in c(t)
        N_deltak1(t-1,1) = ((1-d)*k0+At*exp(theta0).*k0.^alpha-c0.*(1+N_deltac0(t-1,1)))./k1-1; 
                                   % Approximation error in k(t+1)
        N_deltaE0(t-1,1) = 1./(exp1(t-1,1)+exp2(t-1,1)).*  c0.^(-gam).*(1+N_deltac0(t-1,1)).^(-gam)  -1; 
                                   % Approximation error in E(t); see eq. 
                                   % (A9) in JMM (2016)
   end
end
    
% 12. Collecting approximation errors
%------------------------------------
Q_Results = [Q_deltac0 Q_deltak1 Q_deltaE0];
                                   % Results from using the linearized
                                   % model's equations;
                                   % (T_test-1)-by-3
N_Results = [N_deltac0 N_deltak1 N_deltaE0];
                                   % Results from using nonlinear model's
                                   % equations; (T_test-1)-by-3

% 13. Accuracy tests
%-------------------
discard = 0;                       % Discard the first observations to 
                                   % eliminate the effect of initial
                                   % condition

% 13.1 Approximation errors for QUADPROG procedure
%-------------------------------------------------
Q_Mean_all(i,:) = log10(mean(abs(Q_Results(1+discard:end,1:3)))); 
                                   % Mean deltas across all data points                                  
Q_Max_all(i,:) = log10(max(abs(Q_Results(1+discard:end,1:3))));   
                                   % Maximum deltas across all data points                               
Q_Mean_1(i,:) = log10(mean(mean(abs(Q_Results(1+discard:end,1:3))))); 
                                   % Mean deltas across all deltas
Q_Max_1(i,:) = log10(max(max(abs(Q_Results(1+discard:end,1:3)))));
                                   % Maximum delta across all deltas

fprintf(1,'ORDER OF PERTURBATION = %i\n',i);
disp(''); 
fprintf('ALL MEAN DELTAS ACROSS ALL SIMULATED POINTS, QUADPROG \n')
disp('   deltac0   deltak1  deltaE0'); 
disp([Q_Mean_all(i,1) Q_Mean_all(i,2) Q_Mean_all(i,3)]);

fprintf('ONE MEAN DELTA ACROSS ALL DELTAS, QUADPROG \n')
disp('   Mean_delta'); 
disp([Q_Mean_1(i,1)  ]);

fprintf('ALL MAX DELTAS ACROSS ALL SIMULATED POINTS, QUADPROG \n')
disp('   deltac0   deltak1  deltaE0'); 
disp([Q_Max_all(i,1) Q_Max_all(i,2) Q_Max_all(i,3)]);

fprintf('ONE MAX DELTA ACROSS ALL DELTAS, QUADPROG \n')
disp('   Max_delta'); 
disp([Q_Max_1(i,1)  ]);

% 13.2 Approximation errors for FMINSEARCH procedure
%----------------------------------------------
N_Mean_all(i,:) = log10(mean(abs(N_Results(1+discard:end,:)))); 
                                   % Mean deltas across all data points                                  
N_Max_all(i,:) = log10(max(abs(N_Results(1+discard:end,:))));   
                                   % Maximum deltas across all data points                               
N_Mean_1(i,:) = log10(mean(mean(abs(N_Results(1+discard:end,:))))); 
                                   % Mean deltas across all deltas
N_Max_1(i,:) = log10(max(max(abs(N_Results(1+discard:end,:)))));
                                   % Maximum delta across all deltas

fprintf(1,'ORDER OF PERTURBATION = %i\n',i);
disp(''); 
fprintf('ALL MEAN DELTAS ACROSS ALL SIMULATED POINTS, FMINSEARCH \n')
disp('   deltac0   deltak1  deltaE0'); 
disp([N_Mean_all(i,1) N_Mean_all(i,2) N_Mean_all(i,3)]);

fprintf('ONE MEAN DELTA ACROSS ALL DELTAS, FMINSEARCH \n')
disp('   Mean_delta'); 
disp([N_Mean_1(i,1)  ]);

fprintf('ALL MAX DELTAS ACROSS ALL SIMULATED POINTS, FMINSEARCH \n')
disp('   deltac0   deltak1  deltaE0'); 
disp([N_Max_all(i,1) N_Max_all(i,2) N_Max_all(i,3) ]);

fprintf('ONE MAX DELTA ACROSS ALL DELTAS, FMINSEARCH \n')
disp('   Max_delta'); 
disp([N_Max_1(i,1)  ]);

% 13.3 Computing the mean and maximum residuals 
%---------------------------------------------
Residuals = [ResidualsEeq  ResidualsBC];
                                   % Collect the residuals in two FOCs in
                                   % one vector
Mean_2(i,:) = log10(mean(abs(Residuals(1+discard:end,:))));
                                   % Mean residuals for each of 2 equations
Mean_1(i,:) = log10(mean(mean(abs(Residuals(1+discard:end,:)))));
                                   % Mean residuals for both equations
Max_2(i,:) = log10(max(abs(Residuals(1+discard:end,:))));
                                   % Maximum residuals for each of 2 equations
Max_1(i,:) = log10(max(max(abs(Residuals(1+discard:end,:)))));
                                   % Maximum residuals for both equations
 

fprintf(1,'ORDER OF PERTURBATION = %i\n',i);
disp(''); 
fprintf('RESIDUALS \n')
disp('   Mean_Eeq   Mean_BC  Mean_All  Max_Eeq   Max_BC   Max_All'); 
disp([Mean_2(i,1) Mean_2(i,2) Mean_1(i,1) Max_2(i,1) Max_2(i,2) Max_1(i,1)  ]);

fprintf(1,'ORDER OF PERTURBATION = %i\n',i);
disp(''); 
time_PER(i,1) = toc
% save Main_Neocl_gam01;

end

