% "SSE" is a routine for computing the sum of squared approximation errors
% with errors in future variables computed as errors in integration nodes; 
% see the article "Lower Bounds on Approximation Errors to Numerical  
% Solutions of Dynamic Economic Models" by Kenneth L. Judd, Lilia Maliar 
% and Serguei Maliar (2016), Econometrica, forthcoming,(henceforth, JMM, 2016). 
%
%
% This version: February 12, 2017. First version: January 20, 2012.
% -------------------------------------------------------------------------
% Inputs:    "z"            is the unknown (sub)vector of approximation errors;
%                           it includes an error in current consumption 
%                           c(t), deltac(t), and errors in future 
%                           consumption c(t+1) in all integration nodes, 
%                           deltac(t+1,j);
%            "beta"         is the discount factor; 
%            "weights_norm" is the vector of integration weights;
%            "k0"           is the beginning-of-the-period capital stock;
%            "c0"           is current-period consumption;
%            "theta0"       is the current-period productivity level;       
%            "k1"           is the end-of-the-period capital stock;
%            "theta1"       is the next-period productivity level;
%            "c1"           is next-period consumption;
%            "d"            is the depreciation rate;
%            "alpha"        is the capital share in production;
%            "At"           is the normalizing constant in the production
%                           function
%            "gam"          is the coefficient of relative risk aversion
%
% Output:    "y" is         is the sum of squared approximation errors
% -------------------------------------------------------------------------


function [y] = SSE(z,beta,weights_norm,k0,c0,theta0,k1,theta1,c1,d,alpha,At,gam)

n_nodes = size(weights_norm,1); % The number of integration nodes is equal 
                                % to the number of integration weights  
deltac0 = z(1);                 % The first element of vector z is deltac(t)
deltac1 = z(2:n_nodes+1);       % The remaining elements of vector z are
                                % deltac(t+1,j), j=1,...,n_nodes

% Computing the sum of squared approximation errors
%--------------------------------------------------
y = 10000*(     norm((((   (beta*weights_norm'*(c1.^(-gam)./c0.^(-gam).*(1+deltac1)'.^(-gam).*(1-d)))+...
    (  (   (1-d).*k0+exp(theta0).*At.*k0.^alpha-c0.*(1+deltac0)  )./k1   ).^(alpha-1).*...
    (   beta*weights_norm'*(   c1.^(-gam)./c0.^(-gam).*alpha.*At.*exp(theta1).*k1.^(alpha-1).*(1+deltac1)'.^(-gam)    ) )) .^(-1/gam)-1).^2),2)  +  ...                                                             + ...
    norm(((((1-d)*k0+At*exp(theta0).*k0.^alpha-c0.*(1+deltac0))./k1-1).^2),2) + ...  
    norm(deltac1.^2*ones(n_nodes,1),2)).^0.5;
% The first element of the sum is deltac(t)^2 and is given by eq. (A7); the 
% second element is deltak(t+1)^2 and is given by eq. (A6); the last
% element is the vector of deltac(t+1,j), j=1,...,n_nodes

