% "Simulation_Neocl" is a routine that simulates a solution of the
% neoclassical stochastic growth model, produced by perturbation methods,
% implemented in Dynare, that is considered in the article "Lower Bounds on 
% Approximation Errors to Numerical Solutions of Dynamic Economic Models" 
% by Kenneth L. Judd, Lilia Maliar and Serguei Maliar (2016), Econometrica, 
% forthcoming,(henceforth, JMM, 2016). This routine does not use other 
% routines: it computes the decision rules using the matrices of coefficients.
% -------------------------------------------------------------------------
% Inputs:   "SS", "del2", "A", "B", "C", "D", "E" are the matrices of 
%                  coefficients from which the perturbation solution is 
%                  constructed;
%            "epsi_test" is the given vector of current-period shocks;
%            "sigma" is the standard deviation of the productivity shock;
%            "T" is the simulation length
%
% Output:    "PER1", "PER2" are the first-order and second-order
%            perturbation solutions simulated for T periods           
% -------------------------------------------------------------------------
% Copyright  2017 by Lilia Maliar and Serguei Maliar. All rights reserved. 
% The code may be used, modified and redistributed under the terms provided 
% in the file "License_Agreement.txt".
% -------------------------------------------------------------------------


function [PER1, PER2]  = Simulation_Neocl(SS,del2,A,B,C,D,E,epsi_test,sigma,T)

% The variables in the Dynare program "Dynare_neocl.mod" are declared in the 
% following order: k, theta, c; this is also the decision rule (DR) order.
  
PER1 = SS;         % Intialize PER1 by assuming steady state values; 3-by-1
PER2 = SS;         % Intialize PER2 by assuming steady state values; 3-by-1

for t  = 1:T
    DEV_all = PER1(:,t) - SS; 
                   % Compute a deviation of the vector of all variables from 
                   % the steady state, i.e., y(t)-SS
    DEV = DEV_all(1:2); 
                   % Consider y(t)-SS for the state variables, which are k 
                   % and theta (1st and 2d variables in the list)
    EPS = epsi_test(t,1)*sigma; 
                   % Form a matrix of exogenous variables (shocks) multiplied 
                   % by the corresponding standard deviation
    PER1(:,t+1) = SS + A*DEV + B*EPS;
                   % Compute PER1 (see a comment in Section 5.1 of Main_Neocl.m)
end
for t = 1:T
    DEV_all = PER2(:,t)-SS;
                   % Compute a deviation of the vector of all variables from 
                   % the steady state, i.e., y(t)-SS
    DEV = DEV_all(1:2); 
                   % Consider y(t)-SS for the state variables which are k 
                   % and theta (1st and 2d variables in the list)
    EPS = epsi_test(t,1)*sigma;
                   % Form a matrix of exogenous variables (shocks) multiplied 
                   % by the corresponding standard deviation
    DEV2 = kron(DEV,DEV);
                   % Compute a Kronecker product of the vector of state 
                   % variables 
    EPS2 = kron(EPS,EPS);
                   % Compute a Kronecker product of the exogenous variables
    DEVEPS = kron(DEV,EPS);
                   % Compute a Kronecker product of the vector of state
                   % variables by the vector of exogenous variables
    PER2(:,t+1) = SS + A*DEV + B*EPS + 0.5*del2 + 0.5*C*DEV2 + 0.5*D*EPS2 + E*DEVEPS;
                   % Compute PER2 (see a comment in Section 5.2 of Main_Neocl.m)
end