// Dynare routine that solves the new Keynesian  model described in the article 
// "Lower Bounds on Approximation Errors to Numerical Solutions of Dynamic 
// Economic Models" by Kenneth L. Judd, Lilia Maliar and Serguei Maliar (2016), 
// Econometrica, forthcoming,(henceforth, JMM, 2016). 

// This code is based on Eric Smith's (2014) code

// This version: February 12, 2017. First version: January 20, 2012.


// Block 1. Preamble

var pisharp mc Khat G q dY dC dI dN int w Y A N vp K omegag C Z infl I 
lam R u  mu wsharp f1 f2  x1 x2;

varexo ea ez eg ei;

parameters psi beta phip phiw alpha eta b chi1 chi2 tau delta epsw epsp
zetap zetaw rhoi phipi phiy rhog rhoa rhoz omega pistar sigma_ea sigma_ez 
sigma_eg sigma_ei;

// Load the parameters values

load parameterfile;

set_param_value('b', b);
set_param_value('phiw', phiw);
set_param_value('phip', phip);
set_param_value('zetaw', zetaw);
set_param_value('zetap', zetap);
set_param_value('eta', eta);
set_param_value('tau', tau);
set_param_value('rhoi', rhoi);
set_param_value('rhoa', rhoa);
set_param_value('rhoz', rhoz);
set_param_value('rhog', rhog);
set_param_value('phipi', phipi);
set_param_value('phiy', phiy);
set_param_value('psi', psi);
set_param_value('beta', beta);
set_param_value('alpha', alpha);
set_param_value('delta', delta);
set_param_value('chi1', chi1);
set_param_value('chi2', chi2);
set_param_value('epsw', epsw);
set_param_value('epsp', epsp);
set_param_value('pistar', pistar);
set_param_value('omega', omega);
set_param_value('sigma_ea', sigma_ea);
set_param_value('sigma_ez', sigma_ez);
set_param_value('sigma_eg', sigma_eg);
set_param_value('sigma_ei', sigma_ei);


// Block 2. Declaration of the model

model;

 // (1) marginal utility: Eq (B69) in JMM (2016)
 exp(lam) = (exp(C)-b*exp(C(-1)))^(-1)-beta*b*(exp(C(+1))-b*exp(C))^(-1);

 // (2) FOC on utilization: Eq (B70) in JMM (2016)
 exp(R) = exp(Z)^(-1)*(chi1 + chi2*(exp(u)  - 1));

 // (3) Euler equation: Eq (B71) in JMM (2016)
 exp(lam) = beta*exp(lam(+1))*(1+int)*(1+infl(+1))^(-1);

 // (4) FOC on investment: Eq (B72) in JMM (2016)
 exp(lam) = exp(mu)*exp(Z)*(1 - (tau/2)*(exp(I)/exp(I(-1)) - 1)^2 - tau*(exp(I)/exp(I(-1)) - 1)*(exp(I)/exp(I(-1)))) + 
 beta*exp(mu(+1))*exp(Z(+1))*tau*(exp(I(+1))/exp(I) - 1)*(exp(I(+1))/exp(I))^2;

 // (5) FOC on capital: Eq (B73) in JMM (2016)
 exp(mu) = beta*(exp(lam(+1))*(exp(R(+1))*exp(u(+1)) - 
 exp(Z(+1))^(-1)*(chi1*(exp(u(+1)) - 1) + (chi2/2)*(exp(u(+1)) - 1)^2))
 + exp(mu(+1))*(1-delta));

 // (6) Reset wage: Eq (B74) in JMM (2016)
 exp(wsharp) = (epsw/(epsw - 1))*exp(f1)/exp(f2);

 // (7) f1: Eq (B75) in JMM (2016)
 exp(f1) = psi*(exp(w)/exp(wsharp))^(epsw*(1+eta))*exp(N)^(1+eta) +
 phiw*beta*(1+infl)^( - zetaw*epsw*(1+eta))*(1+infl(+1))^(epsw*(1+eta))*
 (exp(wsharp(+1))/exp(wsharp))^(epsw*(1+eta))*exp(f1(+1));

 // (8) f2: Eq (B76) in JMM (2016)
 exp(f2) = exp(lam)*(exp(w)/exp(wsharp))^(epsw)*exp(N) +
 phiw*beta*(1+infl)^(zetaw*(1-epsw))*(1+infl(+1))^(epsw-1)*
 (exp(wsharp(+1))/exp(wsharp))^(epsw)*exp(f2(+1));

 // (9) Real wage index: Eq (B77) in JMM (2016)
 exp(w)^(1-epsw) = (1-phiw)*exp(wsharp)^(1-epsw) +
 (1+infl(-1))^(zetaw*(1-epsw))*(1 + infl)^(epsw-1)*phiw*exp(w(-1))^(1-epsw);

 // (10) Production function: Eq (B78) in JMM (2016)
 exp(Y) = exp(A)*exp(Khat)^(alpha)*exp(N)^(1-alpha)/exp(vp);

 // (11) Price dispersion: Eq (B79) in JMM (2016)
 exp(vp) = (1+infl)^(epsp)*((1-phip)*(1+pisharp)^(-epsp) +
 (1+infl(-1))^(-zetap*epsp)*phip*exp(vp(-1)));

 // (12) Price evolution: Eq (B80) in JMM (2016)
 (1+infl)^(1-epsp) = (1-phip)*(1+pisharp)^(1-epsp) +
 phip*(1+infl(-1))^(zetap*(1-epsp));

 // (13) Reset price: Eq (B81) in JMM (2016)
 1+pisharp = (epsp/(epsp-1))*(1+infl)*exp(x1)/exp(x2);

 // (14) x1: Eq (B82) in JMM (2016)
 exp(x1) = exp(lam)*exp(mc)*exp(Y) +
 phip*beta*(1+infl)^(-zetap*epsp)*(1+infl(+1))^(epsp)*exp(x1(+1));

 // (15) x2: Eq (B83) in JMM (2016)
 exp(x2) = exp(lam)*exp(Y) +
 phip*beta*(1+infl)^(zetap*(1-epsp))*(1+infl(+1))^(epsp-1)*exp(x2(+1));

 // (16) Factor prices: Eq (B84) in JMM (2016)
 exp(w)/exp(R) = ((1-alpha)/alpha)*exp(Khat)/exp(N);

 // (17) Marginal cost: Eq (B85) in JMM (2016)
 exp(w) = exp(mc)*(1-alpha)*exp(A)*exp(Khat)^(alpha)*exp(N)^(-alpha);

 // (18) Taylor rule: Eq (B86) in JMM (2016)
int = (1-rhoi)*(1/beta *(1+pistar)-1) + rhoi*int(-1) +
 (1-rhoi)*(phipi*(infl - pistar) + phiy*(Y - Y(-1))) + ei;
 
 // (19) Aggregate resource: Eq (B87) in JMM (2016)
 exp(Y) = exp(C) + exp(I) + exp(G) + (chi1*(exp(u) - 1) +
 (chi2/2)*(exp(u) - 1)^2)*(exp(K(-1))/exp(Z));

 // (20) Capital accumulation: Eq (B88) in JMM (2016)
 exp(K) = exp(Z)*(1 - (tau/2)*(exp(I)/exp(I(-1)) - 1)^2)*exp(I) +
 (1-delta)*exp(K(-1));

 // (21) Capital services: see JMM (2016)
 exp(Khat) = exp(u)*exp(K(-1));

 // (22) Process for A: Eq (B15) in JMM (2016)
 A = rhoa*A(-1) + ea; //Here, A means log of productivity 

 // (23) Process for Z: Eq (B7) in JMM (2016)
 Z = rhoz*Z(-1) + ez; //Here, Z means log of productivity

 // (24) Government spend: see JMM (2016)
 exp(G) = omegag*exp(Y);

 // (25) Process for omegag: Eq (B20) in JMM (2016)
 omegag = (1-rhog)*omega + rhog*omegag(-1) + eg;

 // (26) Supplementary variable (Hayashis q)
 exp(q) = exp(mu)/exp(lam);

 // (27) Output growth
 dY = Y - Y(-1);

 // (28) Consumption growth
 dC = C - C(-1);

 // (29) Investment growth
 dI = I - I(-1);

 // (30) Hours growth
 dN = N - N(-1);

end;


// Block 3. Solving the model

// Numerical initial conditions for the computation of the deterministic steady state 
initval;
A = 0;
Z = 0;
int = 1/beta *(1+pistar)-1;
//int = (1/beta - 1)*(1+pistar);
N = log(0.5);
K = log(15);
Khat = log(15);
u = 0;
vp = 0;
Y = log(15^(alpha)*0.5^(1-alpha));
C = log(0.6*15^(alpha)*0.5^(1-alpha));
lam = - log(0.6*15^(alpha)*0.5^(1-alpha)) - log(1-b) + log(1-beta*b);
I = log(delta*15);
R = log(1/beta-1+delta);
mc = log(epsp/(epsp-1));
w = log(((1-alpha)/alpha)*30*(1/beta-1+delta));
infl = pistar;
pisharp = pistar;
wsharp = log(((1-alpha)/alpha)*30*(1/beta-1+delta));
f1 = log(psi*0.5^(1+eta)/(1-phiw*beta));
f2 = log((0.5/(0.6*15^(alpha)*0.5^(1-alpha)))/(1-phiw*beta));
mu = -log(0.6*15^(alpha)*0.5^(1-alpha)) - log(1-b) + log(1-beta*b);
dY = 0;
dC = 0;
dI = 0;
dN = 0;

end;


steady; // Calculate the exact steady state values of the endogenous variables


// Specify shocks
shocks;
var ea; stderr sigma_ea;
var ez; stderr sigma_ez;
var eg; stderr sigma_eg;
var ei; stderr sigma_ei;
end;

// The model is solved and simulated with the "stoch_simul" command
stoch_simul(pruning, nograph, noprint, nomoments);