% MATLAB software that computes approximation errors in a new Keynesian
% model, as described in the article "Lower Bounds on Approximation Errors 
% to Numerical Solutions of Dynamic Economic Models" by Kenneth L. Judd, 
% Lilia Maliar and Serguei Maliar (2016), Econometrica, forthcoming,(hence-
% forth, JMM, 2016). 
%
%
% This version: February 12, 2017. First version: January 20, 2012.
% 
% ------------------------------------------------------------------------
% The software uses the following files: 
% ------------------------------------------------------------------------
% 1. "Main_NK.m"                is a main file for computing approximation
%                                errors in the variables and residuals in 
%                                the optimality conditions for the new
%                                Keynesian model; it uses a Dynare routine 
%                                "Dynare_NK.mod" to find a perturbation 
%                                solution
% 2. "Dynare_NK.mod"             is a Dynare routine that produces first-
%                                and second-order perturbation solutions
% 3. "Simul_NK_PER1.m"           simulates a first-order perturbation 
%                                solution of the NK model
% 4. "Simul_NK_PER2.m"           simulates a second-order perturbation 
%                                solution of the NK model
% 5. "Monomials_1.m"             constructs integration nodes and weights 
%                                for an N-dimensional monomial (non-product) 
%                                integration rule with 2N nodes; borrowed 
%                                from The paper "Numerically Stable and Accurate 
%                                Stochastic Simulation Approaches for Solving 
%                                Dynamic Economic Models" by Kenneth L. Judd, 
%                                Lilia Maliar and Serguei Maliar, (2011), 
%                                Quantitative Economics 2/2, 173210, hence-
%                                forth, JMM(2011)
% 5. "Monomials_2.m"             constructs integration nodes and weights 
%                                for an N-dimensional monomial (non-product)  
%                                integration rule with 2N^2+1 nodes; borrowed 
%                                from JMM (2011) 
% -------------------------------------------------------------------------
% Copyright  2017 by Lilia Maliar and Serguei Maliar. All rights reserved. 
% The code may be used, modified and redistributed under the terms provided 
% in the file "License_Agreement.txt".
% -------------------------------------------------------------------------

clear; close all; clc;

% 1. Estimated parameters
% -----------------------
b = 0.72;              % Subsistence-level parameter in the utility function 
phiw = 0.43;           % Calvo parameter in wage setting
phip = 0.71;           % Calvo parameter in price setting
zetaw = 0.38;          % Wage-indexation parameter        
zetap = 0.03;          % Price-indexation parameter 
eta = 1.23;            % Utility-function parameter
tau = 1.87;            % Capital-adjustment-cost parameter
rhoi = 0.79;           % Taylor-rule parameter
rhoa = 0.99;           % Autocorrelation coefficient in the process for A; 
                       % see eq. (B15) in JMM(2016) 
rhoz = 0.90;           % Autocorrelation coefficient in the process for Z; 
                       % see eq. (B17) in JMM(2016) 
rhog = 0.96;           % Autocorrelation coefficient in the process for omegag; 
                       % see eq. (B20) in JMM(2016) 
phipi = 1.35;          % Taylor-rule parameter
phiy = 0.32;           % Taylor-rule parameter
 
% 2. Calibrated parameters
%-------------------------
psi = 2;                % Utility-function parameter
beta = 0.99;            % Subjective discount factor
alpha = 1/3;            % Share of capital in production
delta = 0.025;          % Depreciation rate
chi1 = 1/beta-(1-delta);% Cost-of-utilization parameter
chi2 = 0.01;            % Cost-of-utilization parameter
epsw = 10;              % Elasticity of substitution in the production  
                        % function of intermediate-good producers
epsp = 10;              % Elasticity of substitution in the production 
                        % function of intermediate-good producers
pistar = 0;             % Inflation target
omega = 0.2;            % Steady state level of a government-spending shock

sigma_ea = 0.0074;      % Standard deviation of shock in a process for A; 
                        % see eq. (B15) in JMM(2016)  
sigma_ez = 0.0091;      % Standard deviation of shock in a process for Z; 
                        % see eq. (B17) in JMM(2016) 
sigma_eg = 0.0038;      % Standard deviation of shock in a process for Z; 
                        % see eq. (B20) in JMM(2016) 
sigma_ei = 0.0013;      % Standard deviation of monetary-policy shock in  
                        % the Taylor rule; see eq. (B21) in JMM(2016) 

% 3. Saving the parameters into "parametrfile" that will be used by 
% "Dynare_Neocl.mod"
%------------------------------------------------------------------     
save parameterfile b phiw phip zetaw zetap eta tau rhoi rhoa rhoz rhog phipi phiy psi ...
    beta alpha delta chi1 chi2 epsw epsp pistar omega sigma_ea sigma_ez sigma_eg sigma_ei 

% 4. Computing a Dynare solution using "Dynare_Neocl.mod"
% -------------------------------------------------------
path(path,'4.3.3\matlab') % Add a Dynare path
dynare Dynare_NK.mod;     % Compute the first- and second-order perturbation 
                          % solutions

                          
% 5. Recovering the matrices that are components of the decision rules 
% produced by Dynare 
%---------------------------------------------------------------------
Order = oo_.dr.order_var;  % Check that the order in which the vaiables 
                           % appear in decision rules (DR-order) coincides 
                           % with the order in which the variables are declared
                           
% 5.1 Recovering the matrices relevant for the first-order perturbation 
% solution (PER1)
%----------------------------------------------------------------------
% The first-order approximation has the form y(t)=SS+A*(y(t-1)-SS)+B*u(t),
% where y(t) and u(t) are endogenous and exogenous (i.e., shocks) variables,
% resp.                           
SS    = oo_.dr.ys;         % The steady state values of all variables appearing
                           % in the order in which the endogenous variables 
                           % are declared in the Dynare program (there are 
                           % 3 variables, k, a and c); 30-by-1
A     = oo_.dr.ghx;        % A matrix of coefficients whose rows correspond
                           % to all the endogenous variables in the DR-order,
                           % and whose columns correspond to the state
                           % variables in the DR-order; 30-by-12
B     = oo_.dr.ghu;        % A matrix of coefficients whose rows correspond
                           % to all the endogenous variables in the DR-order,
                           % and whose columns correspond to the exogenous
                           % variables (shocks) in the DR-order; 30-by-4

% 5.2 Recovering the matrices relevant for the second-order perturbation 
% solution (PER2)
%-----------------------------------------------------------------------                          
% The second-order approximation has the form: 
% y(t)=SS+0.5*del2+A(y(t-1)-SS)+B*u(t)+0.5*C*(y(t-1)-SS)*kron*(y(t-1)-SS)+0.5*D*
% *(u(t)*kron*u(t)+E*(y(t-1)-SS)*kron*u(t))

del2  = oo_.dr.ghs2;       % A column vector whose rows correspond to all 
                           % endogenous variables in the DR-order; 3-by-1
C     = oo_.dr.ghxx;       % A matrix of coefficients whose row correspond
                           % to all the endogenous variables in the
                           % DR-order, and whose columns correspond to the
                           % Kronecker product of the vector of state
                           % variables (there are 6 of them) in the DR-order; 
                           % 3-by-4
D     = oo_.dr.ghuu;       % A matrix of coefficients whose row correspond
                           % to all the endogenous variables in the
                           % DR-order, and whose columns correspond to the
                           % Kronecker product of the exogenous variables
                           % (there is 1 of them) in the declaration order; 
                           % 3-by-1
E     = oo_.dr.ghxu;       % A matrix of coefficients whose row correspond
                           % to all the endogenous variables in the
                           % DR-order, and whose columns correspond to the
                           % Kronecker product of the vector of state
                           % variables (there are 2 of them) in the DR-order 
                           % by the vector of exogenous variables (there is
                           % 1 of them) in the declaration order; 3-by-2

% 6. Simulating the time-series solution for the accuracy tests
%--------------------------------------------------------------
T_test = 10200;            % Length of simulations in the test

% 6.1 Constructing the shocks for the simulation of the solution
% --------------------------------------------------------------
ea = randn(T_test+1,1)*sigma_ea;
ez = randn(T_test+1,1)*sigma_ez;
eg = randn(T_test+1,1)*sigma_eg;
ei = randn(T_test+1,1)*sigma_ei;

epsi_test = [ea ez eg ei];

% 6.2 Simulating the first-order perturbation solution
%------------------------------------------------------
PER1_log = Simul_NK_PER1(SS,A,B,epsi_test,sigma_ea,sigma_ez,sigma_eg,sigma_ei,T_test);
                           % We simulate Dynare solutions  "manually" 
                           % instead of using a Dynare simulation routine

% 6.3 Simulating the second-order perturbation solution
%------------------------------------------------------
PER2_log  = Simul_NK_PER2(SS,del2,A,B,C,D,E,epsi_test,sigma_ea,sigma_ez,sigma_eg,sigma_ei,T_test); 

% 6.4 Transforming Dynare solutions from logs into levels
%--------------------------------------------------------
PER1 = exp(PER1_log);        % Transform for the first-order perturbation
PER1(20,:) = PER1_log(20,:); % (Net) inflation was not in logs
PER1(10,:) = PER1_log(10,:); % (Net) interest rate was not in logs

PER2 = exp(PER2_log);        % Transform for the second-order perturbation
PER2(20,:) = PER2_log(20,:); % (Net) inflation was not in logs
PER2(10,:) = PER2_log(10,:); % (Net) interest rate was not in logs
                         
% 7. Integration method for evaluating accuracy 
% ---------------------------------------------
vcv = diag([sigma_ea^2 sigma_ez^2 sigma_eg^2 sigma_ei^2]);
[n_nodes,epsi_nodes,weight_nodes] = Monomials_1(4,vcv);
                             % Monomial integration rule with 2N nodes
                             
%[n_nodes,epsi_nodes,weight_nodes] = Monomials_2(4,vcv);
                             % Monomial integration rule with 2N^2+1 nodes
                           
% 8. Creating an optimization option structure for function quadprog
%-------------------------------------------------------------------
options = optimset('Algorithm','interior-point-convex','LargeScale', 'on', 'Display', 'off'); 
                             % Set the optimization options
% _________________________________________________________________
%
% Approximation errors and residuals of the first- and second-order
% perturbation solutions
% __________________________________________________________________                           
order = 2;                 % The maximum order of perturbation considered

% 9. Initializing the matrices for the results on approximation errors and 
% residuals
%-------------------------------------------------------------------------
Residuals = zeros(T_test-1,20);
                           % There are 20 first-order conditions
Q_delta = zeros(T_test-1,34);
                           % There are 34 endogenous variables in which 
                           % approximation errors are computed
Mean_Residuals_20 = zeros(order,20);                           
Max_Residuals_20 = zeros(order,20);   
Mean_Residuals_1 = zeros(order,1); 
Max_Residuals_1 = zeros(order,1);
Q_Mean_nodes = zeros(order,34);
Q_Max_nodes = zeros(order,34);
Q_Mean_1_nodes = zeros(order,1);
Q_Max_1_nodes = zeros(order,1);                          
time_PER = zeros(order,1);

% 10. Choosing the appropriate perturbation solution 
%---------------------------------------------------
for i = 1:order            % For a Dynare solution of orders "order"; the 
                           % maximum value of "order" is 2,...
    
    if i == 1              % If the considered perturbation solution is of 
                           % the first order, ...
       PER = PER1;         % Use the corresponding simulated time series in 
                           % levels
       PER_log = PER1_log; % Use the corresponding simulated time series in 
                           % logs
    elseif i == 2          % If the considered perturbation solution is of 
                           % the second order, ...
       PER = PER2;         % Use the corresponding simulated time series in 
                           % levels
       PER_log = PER2_log; % Use the corresponding simulated time series in 
                           % logs
    end


% 11. Compute approximation errors, deltas, and residuals for each data point
%----------------------------------------------------------------------------
tic;
for t = 1:T_test;                 % For each given point, ... 
    
       % 11.1 Values of shocks: current values at t and future values at t+1
       % in n_nodes integration nodes  
       %-------------------------------------------------------------------
       A0 = PER_log(13,t);        % A(t) in the Dynare solution is in logs 
       Z0 = PER_log(19,t);        % Z(t) 
       omegag0 = PER_log(17,t);   % omegag(t) 
       ei0 = ei(t);
        
       A1(1:n_nodes,1) = (ones(n_nodes,1)*A0)*rhoa+epsi_nodes(:,1);
                                  % Equation (B15) in JMM (2016)
       Z1(1:n_nodes,1) = (ones(n_nodes,1)*Z0)*rhoz+epsi_nodes(:,2); 
                                  % Equation (B7) in JMM (2016)
       omegag1(1:n_nodes,1) = (1-rhog)*omega + (ones(n_nodes,1)*omegag0)*rhog+epsi_nodes(:,3);
                                  % Equation (B20) in JMM (2016)
       ei1(1:n_nodes,1) = epsi_nodes(:,4);
                                  % Monetary shock is a white noise
       
       A0 = exp(A0);              % Find A(t) in levels
       Z0 = exp(Z0);              % Find Z(t) in levels
       A1 = exp(A1);              % Find A(t+1) in levels
       Z1 = exp(Z1);              % Find Z(t+1) in levels
       
       % 11.2 Choice variables at t
       %---------------------------
       lam0 = PER(22,t);          % lam(t)
       C1 = PER(18,t);            % C(t)
       I1 = PER(21,t);            % I(t)
       mu0 = PER(25,t);           % mu(t)
       u0 = PER(24,t);            % u(t)
       R0 = PER(23,t);            % R(t)
       wsharp0 = PER(26,t);       % wsharp(t)
       f10 = PER(27,t);           % f1(t)
       f20 = PER(28,t);           % f1(t)
       infl1 = PER(20,t);         % pi(t)
       w1 = PER(11,t);            % w(t)
       x10 = PER(29,t);           % x1(t)
       x20 = PER(30,t);           % x2(t)
       Y1 = PER(12,t);            % Y(t)
       N1 = PER(14,t);            % N(t)
       vp1 = PER(15,t);           % vp(t)
       mc0 = PER(2,t);            % mc(t)
       Khat0 = PER(3,t);          % Khat(t)
       G0 = PER(4,t);             % G(t)
       int1 = PER(10,t);          % j (t)
       K1 = PER(16,t);            % K(t+1)
       q0 = PER(5,t);             % q(t)
       pisharp0 = PER(1,t);       % pisharp(t)

       % 11.3 Variables chosen at t+1
       % ----------------------------  
        
       % Simulating the first-order perturbation solution
       %-------------------------------------------------   
     if i == 1
       DEV_all = PER_log(:,t)-SS;
                           % Compute a deviation of the vector of all 
                           % variables from the steady state, i.e., y(t)-SS
       DEV = DEV_all(10:21); 
                           % Consider y(t)-SS for the state variables
         for j = 1:n_nodes;
          EPS = [epsi_nodes(j,1) epsi_nodes(j,2) epsi_nodes(j,3) epsi_nodes(j,4)]';
                           % Form a matrix of exogenous variables (shocks)
          PER_nodes_log(:,j) = SS + A*DEV + B*EPS;
                           % Compute PER1 (see a comment in Section 5.1 above)
         end
         
       % Simulating the second-order perturbation solution in integration 
       % nodes
       %-----------------------------------------------------------------
       elseif i == 2  
       DEV_all = PER_log(:,t)-SS;
                           % Compute a deviation of the vector of all 
                           % variables from the steady state, i.e., y(t)-SS
       DEV = DEV_all(10:21);
                           % Consider y(t)-SS for the state variables
         for j = 1:n_nodes 
          EPS = [epsi_nodes(j,1) epsi_nodes(j,2) epsi_nodes(j,3) epsi_nodes(j,4)]';
                           % Form a matrix of exogenous variables (shocks)
          DEV2 = kron(DEV,DEV);
                           % Compute a Kronecker product of the vector 
                           % of state variables 
          EPS2 = kron(EPS,EPS);
                           % Compute a Kronecker product of the exogenous 
                           % variables
          DEVEPS = kron(DEV,EPS);
                           % Compute a Kronecker product of the vector of state
                           % variables by the vector of exogenous variables 
          PER_nodes_log(:,j) = SS + A*DEV + B*EPS + 0.5*del2 + 0.5*C*DEV2 + 0.5*D*EPS2 + E*DEVEPS;
                           % Compute PER2 (see a comment in Section 5.2 above)
         end
     end
   
        % Computing the next-period variables in the integration nodes
        %-------------------------------------------------------------
        C2 = exp(PER_nodes_log(18,:));      % C(t+1)
        lam1 = exp(PER_nodes_log(22,:));    % lam(t+1)
        I2 = exp(PER_nodes_log(21,:));      % I(t+1)
        mu1 = exp(PER_nodes_log(25,:));     % mu(t+1)
        Z1 = exp(PER_nodes_log(19,:));      % Z(t+1)
        u1 = exp(PER_nodes_log(24,:));      % u(t+1)
        R1 = exp(PER_nodes_log(23,:));      % R(t+1)
        wsharp1 = exp(PER_nodes_log(26,:)); % wsharp(t+1)
        f11 = exp(PER_nodes_log(27,:));     % f1(t+1)
        f21 = exp(PER_nodes_log(28,:));     % f2(t+1)
        infl2 = PER_nodes_log(20,:);        % infl(t+1)
        w2 = exp(PER_nodes_log(11,:));      % w(t+1)
        x11 = exp(PER_nodes_log(29,:));     % x1(t+1)
        x21 = exp(PER_nodes_log(30,:));     % x2(t+1)
 
     % 11.4 Computing approximation errors and residuals for all t>=2
     %---------------------------------------------------------------
     if t>=2;           % t=1 is an initial period; thus, we start from t=2     
        
      infl0 = PER(20,t-1);                  % pi(t-1)
      w0 = PER(11,t-1);                     % w(t-1)
      C0 = PER(18,t-1);                     % C(t-1)
      I0 = PER(21,t-1);                     % I(t-1)
      N0 = PER(14,t-1);                     % N(t-1)
      Y0 = PER(12,t-1);                     % Y(t-1)
      vp0 = PER(15,t-1);                    % vp(t-1)
      int0 = PER(10,t-1);                   % i(t-1)
      K0 = PER(16,t-1);                     % K(t)
     end


        % Computing the residuals in the first-order conditions
        %------------------------------------------------------
        if t>=2; 
    
        %   Eq (1)
        Exp1 = beta*b.*(C2-b*C1).^(-1)*weight_nodes;
        Res1 = 1/lam0 *(  (C1-b*C0)^(-1) - Exp1 ) - 1;
 
        % Eq (2)
        Res2 = 1/R0 * (Z0^(-1)*(chi1 + chi2*(u0 - 1))) -1;

        % Eq (3)
        Exp3 = beta.*lam1.*(1+int1).*(1+infl2).^(-1)*weight_nodes; 
        Res3 = 1/lam0 * Exp3 - 1;
 
        % Eq (4) 
        Exp4 = beta.*mu1.*Z1.*tau.*(I2./I1 - 1).*(I2./I1).^2 *weight_nodes;
        Res4 = 1/lam0 * (      mu0*Z0*(  1 - tau/2*(I1/I0 - 1)^2 - tau*(I1/I0 - 1)*I1/I0  ) + Exp4      )  - 1;

        % Eq (5)
        Exp5 = ( beta.*(lam1.*(R1.*u1 - Z1.^(-1).*(chi1.*(u1 - 1) + (chi2/2).*(u1 - 1).^2))+ mu1.*(1-delta)) )*weight_nodes;
        Res5 = 1/mu0 *  Exp5  -1;

        % Eq (6)
        Res6 = 1/wsharp0 * ( (epsw/(epsw - 1))*f10/f20 ) -1;
 
        % Eq (7)
        Exp7 = beta*(1+infl2).^(epsw.*(1+eta)).*(wsharp1./wsharp0).^(epsw.*(1+eta)).*f11*weight_nodes;
        Res7 = 1/f10 * ( psi*(w1/wsharp0)^(epsw*(1+eta))*N1^(1+eta) + phiw*(1+infl1)^(-zetaw*epsw*(1+eta))*Exp7 ) - 1;

        % Eq (8)
        Exp8 = (1+infl2).^(epsw-1).*(wsharp1./wsharp0).^(epsw).*f21*weight_nodes;
        Res8 = 1/f20 * ( lam0*(w1/wsharp0)^(epsw)*N1 + phiw*(1+infl1)^(zetaw*(1-epsw))*beta*Exp8 ) -1;

        % Eq (9)
        Res9 = 1/w1^(1-epsw) * ( (1-phiw)*wsharp0^(1-epsw) + (1+infl0)^(zetaw*(1-epsw))*(1 + infl1)^(epsw-1)*phiw*w0^(1-epsw) ) -1;

        % Eq (10)
        Res10 = 1/(Y1*vp1) *  A0*Khat0^alpha*N1^(1-alpha)  -1;

        % Eq (11)
        Res11 = 1/(vp1-(1+infl1)^eps*((1-phip)*(1+pisharp0)^(-epsp)) * (1+infl1)^eps*(1+infl0)^(-zetap*epsp)*phip*vp0)  -1; % WAY2: SMALL RESIDUALS

        % Eq (12)
        Res12 = 1/( (1+infl1)^(1-epsp) - phip*(1+infl0)^(zetap*(1-epsp)) *  (1-phip)*(1+pisharp0)^(1-epsp)) -1; % WAY3: SMALL RESIDUALS
    
        % Eq (13)
        Res13 = 1/(1+pisharp0) * (epsp/(epsp-1))*(1+infl1)*x10/x20 -1;
        
        % Eq (14)
        Exp14 = beta.*(1+infl1).^(-zetap.*epsp).*(1+infl2).^(epsp).*x11*weight_nodes;
        Res14 = 1/x10 * (lam0*mc0*Y1 + phip*Exp14) -1;
 
        % Eq (15) 
        Exp15 = beta.*(1+infl1).^(zetap.*(1-epsp)).*(1+infl2).^(epsp-1).*x21*weight_nodes;
        Res15 = 1/x20 * (lam0*Y1 + phip*Exp15) -1;

        %  Eq (16)
        Res16 = R0/w1 * ( ((1-alpha)/alpha)*Khat0/N1) -1;
 
        %  Eq (17)
        Res17 = 1/w1 * (mc0*(1-alpha)*A0*Khat0^(alpha)*N1^(-alpha)) -1;
 
        % Eq (18)
        Res18 = 1/int1 * (  (1-rhoi)*(  1/beta *(1+pistar)- 1)+ rhoi*int0 + (1-rhoi)*(phipi*(infl1 - pistar) + phiy*(  log(Y1) - log(Y0) ) )  - ei0) - 1;
 
        % Eq (19)
        Res19 = 1/Y1 * (C1 + I1 + G0 + (chi1*(u0 - 1) + (chi2/2)*(u0 - 1)^2)*(K0/Z0)) -1;
 
        % Eq (20)
        Res20 = 1/K1 * (Z0*(1 - (tau/2)*(I1/I0 - 1)^2)*I1 + (1-delta)*K0) -1;

        Residuals(t-1,:) = [Res1 Res2 Res3 Res4 Res5 Res6 Res7 Res8 Res9 Res10 Res11 Res12 Res13 Res14 Res15 Res16 Res17 Res18 Res19 Res20];

 
        % Coefficients of the linear equations with approximation errors
        %---------------------------------------------------------------
 
        % Condition 1:
        % lam0 = (C1-b*C0)^(-1)-beta*b*(C2-b*C1)^(-1);
        % a11*delta_lam0+a13*delta_C0+a14*delta_C1+b1=0
        a11 = -lam0;
        Exp1_0 = beta*b.*(C2-b*C1).^(-2)*weight_nodes;
        a13 = -(C1-b*C0)^(-2)*C1-b*C1*Exp1_0;
        Exp1_1 = beta*b.*(C2-b*C1).^(-2).*C1*weight_nodes;
        a14 = Exp1_1;
        for j = 1:n_nodes
             a14_nodes(j) = beta*b*(C2(j)-b*C1).^(-2)*weight_nodes(j);
        end
        b1 = lam0*Res1;
    
        % Condition 2:
        % R0 = Z0^(-1)*(chi1 + chi2*(u0 - 1));
        % a27 * delta_R0+a28*delta_u0+b2=0
        a27 = -1;
        a28 = 1/(R0*Z0)*chi2*u0;
        b2 = Res2; 
        
        % Condition 3: 
        % lam0 = beta*lam1*(1+int1)*(1+infl2)^(-1);        
        % a31*delta_lam0+a32*delta_lam1+a311*delta_infl2+a39*delta_int1+b3=0
        a31 = -1;
        a315 = int1/(1+int1);
        for j = 1:n_nodes
             a32_nodes(j) = weight_nodes(j);
             a311_nodes(j) = -infl2(j)/(1+infl2(j))*weight_nodes(j);
        end
        b3 = log(1+Res3);

        % Condition 4: 
        % lam0 = mu0*Z0*(1 - (tau/2)*(I1/I0) - 1)^2 - tau*(I1/I0 - 1)*(I1/I0)) + beta*mu1*Z1*tau*(I2/I1 - 1)*(I2/I1)^2;
        % a41*delta_lam0+a45*delta_mu0+a46*delta_mu1+a412*delta_I0+a413*delta_I1+b4=0
        a41 = -lam0;
        a45 = mu0*Z0*(2*tau*(I1/I0)-3/2*tau*(I1/I0)^2+tau/2+1);
        for j = 1:n_nodes
        Exp4_1_nodes(j) =  beta*mu1(j)*Z1(j)*tau*(I2(j)/I1)^3 *weight_nodes(j);
        Exp4_2_nodes(j) = beta*mu1(j)*Z1(j)*tau*(I2(j)/I1)^2 *weight_nodes(j);
        a46_nodes(j) = Exp4_1_nodes(j)-Exp4_2_nodes(j);
        a413_nodes(j) = 3*Exp4_1_nodes(j) - 2*Exp4_2_nodes(j);
        end
        Exp4_1 =  beta.*mu1.*Z1.*tau.*(I2./I1).^3 *weight_nodes;
        Exp4_2 = beta.*mu1.*Z1.*tau.*(I2./I1).^2 *weight_nodes;
        a412 = mu0*Z0*(2*tau*(I1/I0)-3*tau*(I1/I0)^2)-3*Exp4_1+2*Exp4_2;
        b4 = -lam0+mu0*Z0-3/2*mu0*Z0*tau*(I1/I0)^2+2*mu0*Z0*tau*(I1/I0)+mu0*Z0*tau/2+Exp4_1-Exp4_2;

        % Condition 5:
        % mu0 = beta*(lam1*(R1*u1 - Z1^(-1)*(chi1*(u1 - 1) + (chi2/2)*(u1 - 1)^2)) + mu1*(1-delta));
        % a52*delta_lam1+a55*delta_mu0+a56*delta_mu1+a514*delta_R1+a533*delta_u1+b5=0;
        q0 = (   (lam1.*(R1.*u1 - Z1.^(-1).*(chi1.*(u1 - 1) + (chi2/2).*(u1 - 1).^2)) + mu1.*(1-delta)).^(-1)  )*weight_nodes;
        for j = 1:n_nodes
            q0_nodes(j) = (   (lam1(j)*(R1(j)*u1(j) - Z1(j)^(-1)*(chi1*(u1(j) - 1) + (chi2/2)*(u1(j) - 1)^2)) + mu1(j)*(1-delta))^(-1)  )*weight_nodes(j);
            qq0_nodes(j) = -lam1(j)*Z1(j)^(-1)*weight_nodes(j);
            a56_nodes(j) = mu1(j)*(1-delta)*weight_nodes(j);
            a514_nodes(j) = q0_nodes(j)*lam1(j)*R1(j)*u1(j)*weight_nodes(j);
            a533_nodes(j) =  q0_nodes(j)*lam1(j)*(1/Z1(j)*(    chi1*(2*u1(j) - 1)+chi2*(u1(j) - 1)*u1(j)   ) - R1(j)*u1(j))*weight_nodes(j); 
            a52_nodes(j) = ( q0_nodes(j)*(lam1(j)*(  1/Z1(j) * chi2*(u1(j) - 1) + R1(j)*u1(j)) )    )*weight_nodes(j);
        end
        a55 = -1;
        b5 = log(1+Res5);

        % Condition 6:
        % wsharp0 = (epsw/(epsw - 1))*f10/f20;
        % a616*delta_f10+a618*delta_f20+a621*delta_wsharp0+b6=0;
        a616 = 1;
        a618 = -1;
        a621 = -1;
        b6 = log(1+Res6);

        % Condition 7:
        %  f10 = psi*(w1/wsharp0)^(epsw*(1+eta))*N1^(1+eta) +
        %        phiw*beta*(1+infl1)^( - zetaw*epsw*(1+eta))*(1+infl2)^(epsw*(1+eta))*
        %        (wsharp1/wsharp0)^(epsw*(1+eta))*f11;
        % a710*delta_ifl1+a711*delta_infl2+a716*delta_f10*a717*delta_f11+a720*delta_w1+a721*delta_wsharp0+a722*delta_wsharp1+a723*delta_N1+b7=0;
        a710 = - zetaw*epsw*(1+eta)*infl1*phiw*beta*(1+infl1)^( - zetaw*epsw*(1+eta)-1)*Exp7;
        Exp7_1 = beta.*(1+infl2).^(epsw.*(1+eta)-1).*(wsharp1./wsharp0).^(epsw.*(1+eta)).*f11.*infl2*weight_nodes;
        a716 = -f10;
        q2 =  phiw*beta*(1+infl1)^( - zetaw*epsw*(1+eta))*Exp7;
        q1 = psi*(w1/wsharp0)^(epsw*(1+eta))*N1^(1+eta)*(1+eta);
        a720 = q1*epsw;
        a721 = -q1*epsw-q2*epsw*(1+eta);
        a723 = q1;
        b7 = Res7*f10;

        for j = 1:n_nodes
            Exp7_nodes(j) = beta*(1+infl2(j))^(epsw*(1+eta))*(wsharp1(j)/wsharp0)^(epsw*(1+eta))*f11(j)*weight_nodes(j);
            Exp7_1_nodes(j) = beta*(1+infl2(j))^(epsw*(1+eta)-1)*(wsharp1(j)/wsharp0)^(epsw*(1+eta))*f11(j)*infl2(j)*weight_nodes(j);
            a711_nodes(j) =  phiw*(1+infl1)^( - zetaw*epsw*(1+eta))*epsw*(1+eta)*Exp7_1_nodes(j); 
            q2_nodes(j) =  phiw*(1+infl1)^( - zetaw*epsw*(1+eta))*Exp7_nodes(j);
            a717_nodes(j) = q2_nodes(j);
            a722_nodes(j) = q2_nodes(j)*epsw*(1+eta);
        end    


        % Condition 8:
        %  f20 = lam0*(w1/wsharp0)^(epsw)*N1 +
        %        phiw*beta*(1+infl1)^(zetaw*(1-epsw))*(1+infl2)^(epsw-1)*
        %        (wsharp1/wsharp0)^(epsw)*f21);
        % a81*delta_lam0+a810*delta_ifl1+a811*delta_infl2+a818*delta_f20*a819*delta_f21+a820*delta_w1+a821*delta_wsharp0+a822*delta_wsharp1+a823*delta_N1+b8=0;
        q3 = lam0*(w1/wsharp0)^(epsw)*N1;
        a81 = q3;
        a810 =  zetaw*(1-epsw)*infl1*phiw*beta*(1+infl1)^(zetaw*(1-epsw)-1)*Exp8;
        for j = 1:n_nodes
        Exp8_1_nodes(j) = beta*(1+infl2(j))^(epsw-2)*(wsharp1(j)/wsharp0)^(epsw)*f21(j)*infl2(j)*weight_nodes(j);
        Exp8_nodes(j) = (1+infl2(j))^(epsw-1)*(wsharp1(j)/wsharp0)^(epsw)*f21(j)*weight_nodes(j);
        a811_nodes(j) = phiw*beta*(1+infl1)^(zetaw*(1-epsw))*(epsw-1)*Exp8_1_nodes(j);
        q4_nodes(j) = phiw*beta*(1+infl1)^(zetaw*(1-epsw))*Exp8_nodes(j);
        a819_nodes(j) = q4_nodes(j);
        a822_nodes(j) = q4_nodes(j)*epsw;
        end
        Exp8_1 = beta.*(1+infl2).^(epsw-2).*(wsharp1./wsharp0).^(epsw).*f21.*infl2*weight_nodes;
        a818 = -f20;
        q4 = phiw*beta*(1+infl1)^(zetaw*(1-epsw))*Exp8;
        a820 = q3*epsw;
        a821 = -q3*epsw-q4*epsw;
        a823 = q3;
        b8 = Res8*f20;

        % Condition 9:
        %  w1^(1-epsw) = (1-phiw)*wsharp0^(1-epsw) +
        %              (1+infl0)^(zetaw*(1-epsw))*(1 + infl1)^(epsw-1)*phiw
        %              *w0^(1-epsw);
        % a910*delta_infl0+a920*delta_w0+a922*delta_infl1+b9=0;
        a910 = (1+infl0)^(zetaw*(1-epsw))*phiw*w0^(1-epsw)*(epsw-1)*infl1*(1 + infl1)^(epsw-2);
        a920 = (1-epsw)*w0^(1-epsw);
        a921 = (1-phiw)*wsharp0^(1-epsw);
        b9 = w1^(1-epsw)*Res9;

        % Condition 10:
        % Y1 = A0*Khat0^(alpha)*N1^(1-alpha)/vp1;
        % a1023*delta_N1+a1024*delta_Y1+a1025*delta_Khat0+a1026*delta_vp1+b10=0;
        a1023 = 1-alpha;
        a1024 = -1;
        a1025 = alpha;
        a1026 = -1;
        b10 = log(1+Res10);

        % Condition 11:
        %  vp1 = (1+infl1)^(epsp)*((1-phip)*(1+pisharp0)^(-epsp) +
        %       (1+infl0)^(-zetap*epsp)*phip*vp0);
        % a1110*delta_infl1+a1126*delta_vp1+a1127*delta_pisharp=0;
        a1110 = epsp*1/(1+infl1)*infl1;
        a1126 = -1;
        a1127 = - (  ((1-phip)*(1+pisharp0)^(-epsp) + (1+infl0)^(-zetap*epsp)*phip*vp0)  )^(-1)*epsp*(1-phip)*(1+pisharp0)^(-epsp-1)*pisharp0;
        b11 = log(1/vp1 * ( (1+infl1)^eps*((1-phip)*(1+pisharp0)^(-epsp) + (1+infl0)^(-zetap*epsp)*phip*vp0) ) ); % b11=log(1+Res11)

        % Condition 12:
        %  (1+infl1)^(1-epsp) = (1-phip)*(1+pisharp0)^(1-epsp) +
        %                     phip*(1+infl0)^(zetap*(1-epsp));
        % a1210*delta_infl1+a1227*pisharp0+b12=0;
        a1210 = -(1-epsp)*(1+infl1)^(-epsp)*infl1;
        a1227 = (1-phip)*(1-epsp)*(1+pisharp0)^(-epsp)*pisharp0;
        b12 = (1+pisharp0)^(1-epsp)*((1+infl1)^(-1+epsp) *((1-phip)*(1+pisharp0)^(1-epsp) + phip*(1+infl0)^(zetap*(1-epsp))) -1); %b12=log(1+Res12)

        % Condition 13:
        %  1+pisharp0 = (epsp/(epsp-1))*(1+infl1)*x10/x20;
        % a1310*delta_infl1+a1327*delta_pisharp0+a1328*delta_x10+a1329*deltax20+b13=0;
        a1310 = pisharp0/(1+pisharp0);
        a1327 = -pisharp0/(1+pisharp0);
        a1328 = 1;
        a1329 = -1;
        b13 = log(1+Res13);

        % Condition 14:
        %  x10 = lam0*mc0*Y1 +
        %        phip*beta*(1+infl1)^(-zetap*epsp)*(1+infl2)^(epsp)*exp(x11);
        % a141*delta_lam0+a1410*delta_infl1+a1411*delta_infl2+a1424*delta_Y1+a1428*delta_x10+a1430*delta_x11+a1434*delta_mc0+b14=0;
        Exp14_1 = beta.*(1+infl2).^zetap.*x11*weight_nodes;
        a141 = lam0*mc0*Y1;
        a1410 = -zetap*epsp*phip*(1+infl1)^(-zetap*epsp-1)*infl1*Exp14_1;
        Exp14_2 = beta.*(1+infl2).^(zetap-1).*infl2.*x11*weight_nodes;
        for j =1:n_nodes
          Exp14_1_nodes(j) = beta*(1+infl2(j))^zetap*x11(j)*weight_nodes(j);
          Exp14_2_nodes(j) = beta*(1+infl2(j))^(zetap-1)*infl2(j)*x11(j)*weight_nodes(j);
          a1411_nodes(j) = phip*(1+infl1)^(-zetap*epsp)*epsp*Exp14_2_nodes(j); 
          a1430_nodes(j) = phip*(1+infl1)^(-zetap*epsp)*Exp14_1_nodes(j);
        end
        a1424 = lam0*mc0*Y1;
        a1428 = -x10;
        a1434 = lam0*mc0*Y1;
        b14 = Res14*x10;

        %  Condition 15:
        %  x20 = lam0*Y1 +
        %        phip*beta*(1+infl1)^(zetap*(1-epsp))*(1+infl2)^(epsp-1)*x21);
        % a151**delta_lam0+a1510*delta_infl1+a1511*delta_infl2+a1524*delta_Y1+a1529*delta_x20+a1431*delta_x21+b15=0;
        Exp15_1 = beta.*(1+infl2).^(epsp-1).*x21*weight_nodes;
        a151 = lam0*mc0*Y1;
        a1510 = zetap*(1-epsp)*phip*(1+infl1)^(zetap*(1-epsp)-1)*infl1*Exp15_1;
        Exp15_2 = beta.*(1+infl2).^(zetap-2).*infl2.*x21*weight_nodes;
        for j = 1:n_nodes
            Exp15_1_nodes(j) = beta*(1+infl2(j))^(epsp-1)*x21(j)*weight_nodes(j);
            Exp15_2_nodes(j) = beta*(1+infl2(j))^(zetap-2)*infl2(j)*x21(j)*weight_nodes(j);
            a1511_nodes(j) = phip*(1+infl1)^(zetap*(1-epsp))*(epsp-1)*Exp15_2_nodes(j);
            a1531_nodes(j) = phip*(1+infl1)^(zetap*(1-epsp))*Exp14_1_nodes(j);
        end
        a1529 = -x20;
        b15 = Res15*x20;

        %  Condition 16:
        % w1/R0 = ((1-alpha)/alpha)*Khat0/N1;
        % a167*delta_R0+a1620*delta_w1+a1623*delta_N1+a1625*delta_Khat+b16=0
        a167 = 1;
        a1620 = -1;
        a1623 = -1;
        a1625 = 1;
        b16 = log(1+Res16);
 
        %  Condition 17:
        %  w1 = mc0*(1-alpha)*A0*Khat0^(alpha)*N1^(-alpha);
        % a1720*delta_w1+a1723*delta_N1+a1734*delta_mc0+b17=0;
        a1720 = -1;
        a1723 = -alpha;
        a1725 = alpha;
        a1734 = 1;
        b17 = log(1+Res17);

        %  Condition 18:
        %  int1 = (1-rhoi)*(1/beta - 1)*(1+pistar) + rhoi*int0 +
        %        (1-rhoi)*(phipi*(infl1 - pistar) + phiy*(Y1 - Y0)) + ei;
        % a1810*delta_infl1+a1815*delta_int1+a1824*delta_Y1+b18=0;
        a1810 = (1-rhoi)*phipi*infl1;
        a1815 = -int1;
        a1824 = (1-rhoi)*phiy;
        b18 = Res18*int1;
     
        %  Condition 19:
        %  Y1 = C1 + I1 + G0 + (chi1*(u0 - 1) +
        %      (chi2/2)*(u0 - 1)^2)*(K0/Z0);
        % a193*delta_C1+a198*delta_u0+a1912*delta_I1+a1924*delta_Y1+a1936*delta_G0+b19=0;
        a193 = C1;
        a198 = chi1*u0*(K0/Z0)+2*chi2*(u0 - 1)*(K0/Z0)*u0;
        a1912 = I1;
        a1924 = -Y1;
        a1936 = G0;
        b19 = Res19*Y1;
 
        %  Condition 20:
        %  K1 = Z0*(1 - (tau/2)*(I1/I0 - 1)^2)*I1 +
        %       (1-delta)*K0;
        % a2012*delta_I1+a2035*delta_K1+b20=0;
        a2012 = I1;
        a2035 = -K1;
        b20 = Res20*K1;
        
        % Quadratic programming: minimizing the sum of squared deltas s.t.  
       % 20 linear constraints 
        %----------------------------------------------------------------
        % Solve the quadratic minimization problem min 0.5*x'*H*x + f'*x 
        % subject to:  A*x <= b; our problem is given by (A5)
        H = eye(22+12*n_nodes,22+12*n_nodes);
        a = zeros(1,n_nodes);
        f = zeros(22+12*n_nodes,1);
   
       Aeq = [a11  a        a13 a14_nodes   0   a          0   0    0     a         0   a         a    0   0   a         0   a         0     0     a         0     0      0     0    0    0   0   a             a     a      0   0  0; 
           0    a         0   a          0   a         a27 a28   0     a         0   a         a    0   0   a         0   a         0     0     a         0     0      0     0    0    0   0   a             a     a      0   0  0; 
           a31 a32_nodes  0   a          0   a          0   0    0   a311_nodes  0   a         a a315   0   a         0   a         0     0     a         0     0      0     0    0    0   0   a             a     a      0   0  0;
           a41  a         0   a          a45 a46_nodes  0   0    0     a       a412 a413_nodes a    0   0   a         0   a         0     0     a         0     0      0     0    0    0   0   a             a     a      0   0  0;
           0   a52_nodes  0   a          a55 a56_nodes  0   0    0     a         0   a a514_nodes   0   0   a         0   a         0     0     a         0     0      0     0    0    0   0   a             a a533_nodes 0   0  0;
           0    a         0   a          0   a          0   0    0     a         0   a         a    0 a616  a       a618  a         0    a621   a         0     0      0     0    0    0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0   a710 a711_nodes  0   a         a    0 a716 a717_nodes 0   a         a720 a721 a722_nodes a723   0      0     0    0    0   0   a             a     a      0   0  0;
           a81  a         0   a          0   a          0   0   a810 a811_nodes  0   a         a    0   0   a       a818 a819_nodes a820 a821 a822_nodes a823   0      0     0    0    0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0   a910   a         0   a         a    0   0   a         0   a         a920 a921   a         0     0      0     0    0    0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0    0     a         0   a         a    0   0   a         0   a         0     0     a        a1023 a1024 a1025 a1026  0    0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0  a1110   a         0   a         a    0   0   a         0   a         0     0     a          0     0     0   a1126 a1127 0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0  a1210   a         0   a         a    0   0   a         0   a         0     0     a          0     0     0     0   a1227 0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0  a1310   a         0   a         a    0   0   a         0   a         0     0     a          0     0     0     0   a1327 0   0   a             a     a      0   0  0;
         a141   a         0   a          0   a          0   0  a1410 a1411_nodes 0   a         a    0   0   a         0   a         0     0     a          0   a1424   0     0    0 a1428  0 a1430_nodes     a     a   a1434  0  0;
         a151   a         0   a          0   a          0   0  a1510 a1511_nodes 0   a         a    0   0   a         0   a         0     0     a          0     0     0     0    0    0 a1529 a   a1531_nodes     a      0   0  0;
           0    a         0   a          0   a         a167 0    0     a         0   a         a    0   0   a         0   a        a1620  0     a         a1623  0   a1625   0    0    0   0   a             a     a      0   0  0;
           0    a         0   a          0   a          0   0    0     a         0   a         a    0   0   a         0   a        a1720  0     a         a1723  0   a1725   0    0    0   0   a             a     a   a1734  0  0;
           0    a         0   a          0   a          0   0  a1810   a         0   a         a a1815  0   a         0   a         0     0     a          0   a1824   0     0    0    0   0   a             a     a      0   0  0;
           0    a       a193  a          0   a          0 a198   0     a      a1912  a         a    0   0   a         0   a         0     0     a          0   a1924   0     0    0    0   0   a             a     a      0   0 a1936;
           0    a         0   a          0   a          0   0    0     a      a2012  a         a    0   0   a         0   a         0     0     a          0     0     0     0    0    0   0   a             a     a      0 a2035 0];
           
        beq = [b1; b2; b3; b4; b5; b6; b7; b8; b9; b10; b11; b12; b13; b14; b15; b16; b17; b18; b19; b20]; % 20 linear constraints
        warning('off','all');
        [Qx,fval,exitflag,output,lambda] = quadprog(H,f,[],[],Aeq,beq,-1e-3,1e+3,[],options);
                                   % Solve the quadratic minimization 
                                   % problem (B109)

% Naming all errors the (current and future across nodes)
%--------------------------------------------------------
Q_delta_lam0_nodes(t-1,1) = Qx(1,1);
    for j = 1:n_nodes
        Q_delta_lam1(t-1,j,1) = Qx(1+j,1);                     % Future #1
        Q_delta_C1_nodes(t-1,j,1) = Qx(2+n_nodes+j,1);         % Future #2
        Q_delta_mu1_nodes(t-1,j,1) = Qx(3+2*n_nodes+j,1);      % Future #3
        Q_delta_infl2_nodes(t-1,j,1) = Qx(6+3*n_nodes+j,1);    % Future #4
        Q_delta_I1_nodes(t-1,j,1) = Qx(7+4*n_nodes+j,1);       % Future #5
        Q_delta_R1_nodes(t-1,j,1) = Qx(7+5*n_nodes+j,1);       % Future #6
        Q_delta_f11_nodes(t-1,j,1) = Qx(9+6*n_nodes+j,1);      % Future #8
        Q_delta_f21_nodes(t-1,j,1) = Qx(10+7*n_nodes+j,1);     % Future #9
        Q_delta_wsharp1_nodes(t-1,j,1) = Qx(12+8*n_nodes+j,1); % Future #10
        Q_delta_x11_nodes(t-1,j,1) = Qx(19+9*n_nodes+j,1);     % Future #11
        Q_delta_x21_nodes(t-1,j,1) = Qx(19+10*n_nodes+j,1);    % Future #12
        Q_delta_u1_nodes(t-1,j,1) = Qx(19+11*n_nodes+j,1);     % Future #13
    end
    
    Q_delta_C0_nodes(t-1,1) = Qx(2+n_nodes,1);
    Q_delta_mu0_nodes(t-1,1) = Qx(3+2*n_nodes,1);
    Q_delta_R0_nodes(t-1,1) = Qx(4+3*n_nodes,1);
    Q_delta_u0_nodes(t-1,1) = Qx(5+3*n_nodes,1);
    Q_delta_infl1_nodes(t-1,1) = Qx(6+3*n_nodes,1);
    Q_delta_I0_nodes(t-1,1) = Qx(7+4*n_nodes,1);  
    Q_delta_int1_nodes(t-1,1) = Qx(8+6*n_nodes,1);        
    Q_delta_f10_nodes(t-1,1) = Qx(9+6*n_nodes,1);               
    Q_delta_f20_nodes(t-1,1) = Qx(10+7*n_nodes,1);
    Q_delta_w1_nodes(t-1,1) = Qx(11+8*n_nodes,1);
    Q_delta_wsharp0_nodes(t-1,1) = Qx(12+8*n_nodes,1);
    Q_delta_N1_nodes(t-1,1) = Qx(13+9*n_nodes,1);
    Q_delta_Y1_nodes(t-1,1) = Qx(14+9*n_nodes,1);
    Q_delta_Khat_nodes(t-1,1) = Qx(15+9*n_nodes,1);
    Q_delta_vp1_nodes(t-1,1) = Qx(16+9*n_nodes,1);
    Q_delta_pisharp_nodes(t-1,1) = Qx(17+9*n_nodes,1);
    Q_delta_x10_nodes(t-1,1) = Qx(18+9*n_nodes,1);
    Q_delta_x20_nodes(t-1,1) = Qx(19+9*n_nodes,1);
    Q_delta_mc0_nodes(t-1,1) = Qx(20+12*n_nodes,1);
    Q_delta_K1_nodes(t-1,1) = Qx(21+12*n_nodes,1);
    Q_delta_G0_nodes(t-1,1) = Qx(22+12*n_nodes,1);
    
    % Creating a matrix with all errors
    %--------------------------------
    Q_delta_nodes(t-1,:) = [Q_delta_lam0_nodes(t-1,1) Q_delta_lam1(t-1,:,1) Q_delta_C0_nodes(t-1,1) Q_delta_C1_nodes(t-1,:,1) Q_delta_mu0_nodes(t-1,1) Q_delta_mu1_nodes(t-1,:,1) Q_delta_R0_nodes(t-1,1) ...
       Q_delta_u0_nodes(t-1,1) Q_delta_infl1_nodes(t-1,1) Q_delta_infl2_nodes(t-1,:,1) Q_delta_I0_nodes(t-1,1) Q_delta_I1_nodes(t-1,:,1) Q_delta_R1_nodes(t-1,:,1) Q_delta_int1_nodes(t-1,1) Q_delta_f10_nodes(t-1,1) Q_delta_f11_nodes(t-1,:,1) Q_delta_f20_nodes(t-1,1) ...
       Q_delta_f21_nodes(t-1,:,1) Q_delta_w1_nodes(t-1,1)  Q_delta_wsharp0_nodes(t-1,1) Q_delta_wsharp1_nodes(t-1,:,1) Q_delta_N1_nodes(t-1,1) Q_delta_Y1_nodes(t-1,1) Q_delta_Khat_nodes(t-1,1)  Q_delta_vp1_nodes(t-1,1) Q_delta_pisharp_nodes(t-1,1) Q_delta_x10_nodes(t-1,1) ...
       Q_delta_x20_nodes(t-1,1) Q_delta_x11_nodes(t-1,:,1)  Q_delta_x21_nodes(t-1,:,1) Q_delta_u1_nodes(t-1,:,1) Q_delta_mc0_nodes(t-1,1) Q_delta_K1_nodes(t-1,1) Q_delta_G0_nodes(t-1,1)];

    % Creating a new matrix of errors where future errors appear not like a
    % vector (across nodes) but like a mean
    %----------------------------------------------------------------------
    Q_delta_nodes_mean(t-1,:) = [Q_delta_lam0_nodes(t-1,1) mean(abs(Q_delta_lam1(t-1,:,1))) Q_delta_C0_nodes(t-1,1) mean(abs(Q_delta_C1_nodes(t-1,:,1))) ...
        Q_delta_mu0_nodes(t-1,1) mean(abs(Q_delta_mu1_nodes(t-1,:,1))) Q_delta_R0_nodes(t-1,1) Q_delta_u0_nodes(t-1,1) Q_delta_infl1_nodes(t-1,1) ...
        mean(abs(Q_delta_infl2_nodes(t-1,:,1))) Q_delta_I0_nodes(t-1,1) mean(abs(Q_delta_I1_nodes(t-1,:,1))) mean(abs(Q_delta_R1_nodes(t-1,:,1)))  ...
        Q_delta_int1_nodes(t-1,1) Q_delta_f10_nodes(t-1,1) mean(abs(Q_delta_f11_nodes(t-1,:,1))) Q_delta_f20_nodes(t-1,1) mean(abs(Q_delta_f21_nodes(t-1,:,1))) ...
        Q_delta_w1_nodes(t-1,1)  Q_delta_wsharp0_nodes(t-1,1) mean(abs(Q_delta_wsharp1_nodes(t-1,:,1))) Q_delta_N1_nodes(t-1,1) Q_delta_Y1_nodes(t-1,1) ...
        Q_delta_Khat_nodes(t-1,1)  Q_delta_vp1_nodes(t-1,1) Q_delta_pisharp_nodes(t-1,1) Q_delta_x10_nodes(t-1,1) Q_delta_x20_nodes(t-1,1) ...
        mean(abs(Q_delta_x11_nodes(t-1,:,1)))  mean(abs(Q_delta_x21_nodes(t-1,:,1))) mean(abs(Q_delta_u1_nodes(t-1,:,1))) Q_delta_mc0_nodes(t-1,1) ...
        Q_delta_K1_nodes(t-1,1) Q_delta_G0_nodes(t-1,1)];
    
    % Creating a new matrix of errors where future errors appear not like a
    % vector (across nodes) but like a max
    %----------------------------------------------------------------------
    Q_delta_nodes_max(t-1,:) = [Q_delta_lam0_nodes(t-1,1) max(abs(Q_delta_lam1(t-1,:,1))) Q_delta_C0_nodes(t-1,1) max(abs(Q_delta_C1_nodes(t-1,:,1))) ...
        Q_delta_mu0_nodes(t-1,1) max(abs(Q_delta_mu1_nodes(t-1,:,1))) Q_delta_R0_nodes(t-1,1) Q_delta_u0_nodes(t-1,1) Q_delta_infl1_nodes(t-1,1) ...
        max(abs(Q_delta_infl2_nodes(t-1,:,1))) Q_delta_I0_nodes(t-1,1) max(abs(Q_delta_I1_nodes(t-1,:,1))) max(abs(Q_delta_R1_nodes(t-1,:,1)))  ...
        Q_delta_int1_nodes(t-1,1) Q_delta_f10_nodes(t-1,1) max(abs(Q_delta_f11_nodes(t-1,:,1))) Q_delta_f20_nodes(t-1,1) max(abs(Q_delta_f21_nodes(t-1,:,1))) ...
        Q_delta_w1_nodes(t-1,1)  Q_delta_wsharp0_nodes(t-1,1) max(abs(Q_delta_wsharp1_nodes(t-1,:,1))) Q_delta_N1_nodes(t-1,1) Q_delta_Y1_nodes(t-1,1) ...
        Q_delta_Khat_nodes(t-1,1)  Q_delta_vp1_nodes(t-1,1) Q_delta_pisharp_nodes(t-1,1) Q_delta_x10_nodes(t-1,1) Q_delta_x20_nodes(t-1,1) ...
        max(abs(Q_delta_x11_nodes(t-1,:,1)))  max(abs(Q_delta_x21_nodes(t-1,:,1))) max(abs(Q_delta_u1_nodes(t-1,:,1))) Q_delta_mc0_nodes(t-1,1) ...
        Q_delta_K1_nodes(t-1,1) Q_delta_G0_nodes(t-1,1)];

 
  end
end

% 13. Accuracy tests
%-------------------
discard = 200;                     % Discard the first observations to 
                                   % eliminate the effect of initial
                                   % condition
fprintf(1,'ORDER OF PERTURBATION = %i\n',i);                                   
     % 13.1 Residuals
     %---------------
     Results_Residuals = [Residuals(:,1) Residuals(:,2) Residuals(:,3) Residuals(:,4) Residuals(:,5) Residuals(:,6) ...
            Residuals(:,7) Residuals(:,8) Residuals(:,9) Residuals(:,10) Residuals(:,11) Residuals(:,12) ...
            Residuals(:,13) Residuals(:,14) Residuals(:,15) Residuals(:,16) Residuals(:,17) Residuals(:,18) ...
            Residuals(:,19) Residuals(:,20)];
 
     RR_dummy = (sum((imag(Results_Residuals)==0),2)==20)*ones(1,20);
     Results_Residuals1 = Results_Residuals.*RR_dummy;
         % Residuals collected in one matrix; T-by-20
     Mean_Residuals_20(i,:) = log10(mean(abs(Results_Residuals1(1+discard:end,:))));
         % Mean residuals for each equation
     Mean_Residuals_1(i,:) = log10(mean(mean(abs(Results_Residuals1(1+discard:end,:)))));
         % Mean residuals across 6 equations
     Max_Residuals_20(i,:) = log10(max(abs(Results_Residuals1(1+discard:end,:))));
         % Maximum residuals for each equation
     Max_Residuals_1(i,:) = log10(max(max(abs(Results_Residuals1(1+discard:end,:))))); 
         % Maximum residuals across 6 equations
     disp('Table 1: Mean and maximum of residuals across all simulated points'); Table1(1:2,1:21) = [Mean_Residuals_20(i,:) Mean_Residuals_1(i,:); Max_Residuals_20(i,:) Max_Residuals_1(i,:)]

 
     % 13.2 Approximation errors
     %---------------------------
     % Deltas collected in one matrix; T-by-118
     Q_Mean_nodes(i,:) = log10(mean(abs(Q_delta_nodes_mean(1+discard:end,:)))); 
     % Mean deltas for each equation
     Q_Max_nodes(i,:) = log10(max(abs(Q_delta_nodes_max(1+discard:end,:)))); 
     % Max delta for each equation
     Q_Mean_1_nodes(i,:) = log10(mean(mean(abs(Q_delta_nodes_mean(1+discard:end,:)))));
     % Mean delta (one for all equation)
     Q_Max_1_nodes(i,:) = log10(max(max(abs(Q_delta_nodes_max(1+discard:end,:)))));
     % Max delta (one for all equation)
     disp('Table 2: Mean and maximum of deltas across all simulated points, integration nodes and those of all deltas'); Table2 = [Q_Mean_nodes(i,:) Q_Mean_1_nodes(i,:); Q_Max_nodes(i,:) Q_Max_1_nodes(i,:)]
     
     
     fprintf(1,'ORDER OF PERTURBATION = %i\n',i);
     disp(''); 
     time_PER(i,1) = toc

end 

% save NK_sigma_Est 

