% "Simul_NK_PER1" is a routine that simulates a second-order perturbation 
% solution of the new Keynesian model, produced by Dynare, that is considered 
% in the article "Lower Bounds on Approximation Errors to Numerical Solutions 
% of Dynamic Economic Models" by Kenneth L. Judd, Lilia Maliar and Serguei  
% Maliar (2016), Econometrica, forthcoming,(henceforth, JMM, 2016). This 
% routine does not use other routines: it computes the decision rules using 
% the matrices of coefficients.
% -------------------------------------------------------------------------
% Inputs:   "SS", "del2", "A", "B", "C", "D", "E" are the matrices of 
%                  coefficients from which the perturbation solution is 
%                  constructed;
%            "epsi" is the given vector of current-period shocks;
%            "sigma_ea", "sigma_ez", "sigma_eg", "sigma_ei" are the standard 
%                   deviations of shocks to variables A, Z, omegag and i
%            "T" is the simulation length
%
% Output:    "PER1", "PER2" are the first-order and second-order
%            perturbation solutions simulated for T periods           
% -------------------------------------------------------------------------
% Copyright  2017 by Lilia Maliar and Serguei Maliar. All rights reserved. 
% The code may be used, modified and redistributed under the terms provided 
% in the file "License_Agreement.txt".
% -------------------------------------------------------------------------


function [PER2]  = Simul_NK_PER2(SS,del2,A,B,C,D,E,epsi,sigma_ea,sigma_ez,sigma_eg,sigma_ei,T)

% The variables in the Dynare program "Dynare_NK.mod" are declared in the 
% following order: : pisharp mc Khat G q dY dC dI dN int w Y A N vp K omegag 
% C Z infl I lam R u  mu wsharp f1 f2  x1 x2 

PER2 = SS;           % Initialize PER2 by assuming steady state values; 30-by-1

for t = 1:T
    DEV_all = PER2(:,t)-SS; 
                   % Compute a deviation of the vector of all variables from 
                   % the steady state, i.e., y(t)-SS
    DEV = DEV_all(10:21);
                   % Consider y(t)-SS for the state variables, which are k 
                   % and theta (1st and 2d variables in the list)
    EPS = [epsi(t,1)*sigma_ea epsi(t,2)*sigma_ez epsi(t,3)*sigma_eg epsi(t,4)*sigma_ei]';
                   % Form a matrix of exogenous variables (shocks) multiplied 
                   % by the corresponding standard deviation
    DEV2 = kron(DEV,DEV);
                   % Compute a Kronecker product of the vector of state 
                   % variables 
    EPS2 = kron(EPS,EPS);
                   % Compute a Kronecker product of the exogenous variables
    DEVEPS = kron(DEV,EPS);
                   % Compute a Kronecker product of the vector of state
                   % variables by the vector of exogenous variables
    PER2(:,t+1) = SS+A*DEV+B*EPS+0.5*del2+0.5*C*DEV2+0.5*D*EPS2+E*DEVEPS;
                   % Compute PER2 (see a comment in Section 5.2 of Main_NK.m)
end
