%======================================================
%  Matlab Demonstration of online bidding mechanism of 
%  the World-Cup auction example the future value function
%   u(s)=(1/m)u(s)  (m: number of states) for each state
%
% Input:
%(ak,pik,qk): the kth bidder's submission
% pp: the current price for each of m states (initially set to 1/m)
% bb: the outstabding shares of each state (initially set to 0)
%   
% Output:
% xk: the shares granted to the kth bidder
% pp: new price for each state
% bb: new outstabding shares for each state
%
%  Details can be found from Example 2.7 in Sect. 2.2 of
%  L&Y, Linear and nonlinear programming, 5th edition
%======================================================%
function [xk,pp,bb] = onlineWorldcup(ak,pik,qk,pp,bb);
[m,n]=size(ak);
mu=1/m;
%
% Step 1
%
 if ak'*pp > pik, disp('Bid price too low');
    xk=0
    return;
 end;
%
% next step
% Set data
 [m,n] = size(pp);
 ee=ones(m,1);
 m = m+1;
 A=[[ak -ee; 1 0] speye(m)];
 b=[-bb;qk];
 n = m+3;
% Set parameters
 toler=1.e-6; 
 gamma=.5;     
 alpha=0.7;    
%
% Set initial points
%
 x = ones(n,1);
 s = ones(n,1);
 y = zeros(m,1);
 mu1= n;
 mu2= 1;
 gap = 1;
 iter =0;
 dr0  =1.e10;
%
% Start the loop
%
 while gap >= toler,
   iter = iter + 1;
   %
   % Get objective gradient and hessian
   %
   xx = x(1:n-1)/x(n);
   ss = xx(3:m+1);
   % gradient
   % qq = [-pik;1;-mu*(ee./(1+ss));0]; %u(s)=log(s);
   qq = [-pik;1;-mu*(ee./(1+ss));0]; %u(s)=log(1+s);
   % hessian
   % hh = [0;0;mu*(ee./(ss.*ss));0]; %u(s)=log(s)
   hh = [0;0;mu*(ee./((1+ss).*(1+ss)));0]; %u(s)=log(1+s)
   MM = sparse(diag(hh));
   %
   % Form the homoginized gradient and hessian
   %
   MM = [MM qq-MM*xx;-qq'-xx'*MM xx'*MM*xx];
   qq = [x(n)*qq-A'*y-s(1:n-1);-qq'*x(1:n-1)+b'*y-s(n)];
   %
   % Solving one Newton step with the augmented system
   %
   XX=sparse(1:n,1:n,x/mu2);
   SS=sparse(1:n,1:n,s/mu2);
   gamma = min((mu2/mu1)^2,.5); % Choose centering parameter
   %
   % Check dual feasibility residual
   dr = norm(qq);
   if (dr > 2*dr0)
       %
       % If dual feasibility is not improving, set Newton step to only imprve
       % dual equation while keeping primal feasibility and dualty gap unchanged
       gamma=0;
       qqq = [ones(n,1)-x.*s/mu2-XX*qq;sparse(m,1)];
   else
       % Otherwise, set Newton step to improve all of them at same time
       qqq = [gamma*ones(n,1)-x.*s/mu2-(1-gamma)*XX*qq;-(1-gamma)*(A*x(1:n-1)-b*x(n))];
   end;
   dr0=dr;
   %
   % Linear system solve
   dx=[XX*MM+SS -XX*[A -b]';[A -b] sparse(m,m)]\qqq;
   %
   % Construct primal and dual steps
   dy=dx(n+1:n+m);
   dx=dx(1:n);
   ds=MM*dx-[A -b]'*dy+(1-gamma)*qq;
%
%  choose step-size
%
   nora = min([dx./x;ds./s]);
   nora = abs(alpha/nora);
%
% Update iterates
%
   x = x + nora*dx;
   s = s + nora*ds;
   y = y + nora*dy;
%
% Recompute duality gap
%
   mu1 = mu2;
   mu2 = x'*s/n;
   gap = mu2;
 end;
%
% Output solution or infeasibility certificate
%
%iter
 n = n-1;
 if s(n+1) < x(n+1),
   y=y/x(n+1);
   x=x(1:n)/x(n+1);
   s=[-pik;1;-mu*(ee./x(3:m+1));0]-A'*y;
   pp=-y(1:m-1);
   xk=x(1);
   bb=bb+xk*ak;
   s=max(sparse(n,1),s);
   disp('Bid successful:');
   xk
 else
   x=x(1:n)/s(n+1);
   y=y/s(n+1);
   s=s(1:n)/s(n+1);
   if (b'*y > .5);
      disp('The problem is (near) infeasible');
   else
      disp('The problem is (near) unbounded');
   end;
 end;
 return
%



