% Newton's method for solving the HSDLP system after 
% dual-substitution to replace x from center equations Sx=mu*e
%
%    mu A*S^{-1}e  -b*tau    = 0
%   -A'*y      +c*tau -s     = 0
%   b'*y -c'*x       -mu/tau = 0
%  
% Output
%     x>=0   : approximate primal solution or infeasibility
%              certificate
%     s>=0   : dual slack variables
%              x^Ts =< toler.
%     y      : multipliers of equality constraints or 
%              infeasibility certificate
%     iter   : number of iterations
%
% Zero-feasibility strategy Details can be found in Sect. 
% 5.7 of L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [x,y,s,iter] = DualHOlptau(A,b,c);
% Set parameters
%
 [m,n] = size(A);
 toler=1.e-4; 
 alpha=0.7; % Need a step-size to keep s and tau positive
 gamma=0.8;
%
% Set initial solutions
%
 ee= ones(n,1);
 s = ee;
 tau=1;
 y = zeros(m,1);
 mu = 1;
% Set initial residuals
 rp0=A*ee-b;
 rd0=c-ee;
 r00=-c'*ee-1;
 iter =0;
 tic;
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   % Reduce mu by a factor
   mu = gamma*mu;
   % Scale input data
   D = sparse(1:n,1:n,1./s);
   AD=A*D;
   Dc=D*c;
   %
   % Set up residuals and scaling matrix
   %
   rp=AD*(mu*ee)-tau*b;
   rd= tau*c-A'*y-s;
   r0= b'*y-mu*(Dc'*ee)-mu/tau;
   %
   % Apply Newton step
   %
   % Set up linear system for directions dy and dtau
   ADDc=AD*Dc;
   Drd=D*rd;
   % Solve dy and dtau together
   %rr = [-rp+mu*(AD*Drd);-r0-mu*(Dc'*Drd)];
   %dtau=[mu*(AD*AD') -b-mu*ADDc;b'- mu*ADDc' mu*(1/tau^2+Dc'*Dc)]\rr;
   %dy = dtau(1:m);
   %dtau=dtau(m+1);
   % Solve dy and dtau seperately
   rry=-rp+mu*(AD*Drd);
   rrt=-r0-mu*(Dc'*Drd);
   b1 =b+mu*ADDc;
   b2 =b-mu*ADDc;
   (AD*AD')\[rry b1]/mu;
   dy=ans(:,1);
   dy1=ans(:,2);
   dd=b2'*dy1+mu*(1/tau^2+Dc'*Dc);
   if abs(dd) < toler^3,
       dtau=0;
   else
       dtau=(rrt-b2'*dy)/dd;
   end;
   dy = dy+dtau*dy1;
   %
   ds=rd-A'*dy+dtau*c;
%
%  choose step-size to keep s and tau positive
%
   nora = min([dtau/tau;ds./s]);
   if nora < 0,
      nora = min(1,abs(alpha/nora));
   else
      nora=1;
   end;
%
% Update iterates
%
   y = y + nora*dy;
   s = s + nora*ds;
   tau = tau + nora*dtau;
%
 end;
 % Recover the primal solution if needed
 D = sparse(1:n,1:n,1./s);
 AD= A*D;
 yy=(AD*AD')\(b*tau-mu*(AD*(D*ee)));
 x=D^2*(mu*ee+A'*yy);
 kappa=mu/tau;
 % Check feasible or infeasible case and output final solutions
 if tau>kappa,
     x=x/tau;
     y=y/tau;
     s=s/tau;
 else
     x=x/kappa;
     y=y/kappa;
     s=s/kappa;
     disp('The problem is either infeasible or unbounded.');
 end;
 time=toc
%