%  Primal-Dual scaling for HSDLP by Newton's method for solving
%
%         A*x  -b*tau = 0
%   -A'*y      +c*tau -s = 0
%    b'*y-c'*x        -kappa=0
%        x.*s - mu*1 =0
%      tau*kappa-mu  =0
%  
%  Output
%     x>=0   : approximate primal solution,
%     s>=0   : dual slack variables
%     y      : multipliers of equality constraints
%              x^Ts =< toler.
%     tau>=0,kappa>=0: tau>>kappa implies feasible
%     iter   : number of iterations
%
%   Details can be found in Sect. 5.7 of
%   L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [x,y,s,tau,kappa,iter] = PrimalDualHOlptaukappa(A,b,c);
% Set parameters
%
 [m,n] = size(A);
 toler=1.e-4; 
 alpha=0.5;
 gamma=0.8;
%
% Set initial solutions
%
 ee= ones(n,1);
 x = ee;
 s = ee;
 tau=1;
 kappa=1;
 y = zeros(m,1);
 mu = 1;
 iter =0;
 tic;
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   Ds = sparse(1:n,1:n,1./s);
   Dxs= sparse(1:n,1:n,x./s);
   ADs=A*Ds;
   ADxs=A*Dxs;
   ADxsc=ADxs*c;
   Dxsc=Dxs*c;
   %
   % Set up residuals and scaling matrix
   %
   rp=A*x-tau*b;
   rd= tau*c-A'*y-s;
   r0= b'*y-c'*x-kappa;
   rmu=x.*s-mu*ee;
   rtk=tau*kappa-mu;
   %
   % Newton step
   %
   % Solve linear sustem for dy and dtau together
   %rr = [-rp+ADs*rmu+ADxs*rd;-r0-rtk/tau-Dxsc'*rd-c'*Ds*rmu];
   %d=[ADxs*A' -b-ADxsc;b'- ADxsc' kappa/tau+c'*Dxsc]\rr;
   %dy=d(1:m);
   %dtau=d(m+1);
   % Solve linear system for dy and dtau seperately
   rry=-rp+ADs*rmu+ADxs*rd;
   rrt=-r0-rtk/tau-Dxsc'*rd-c'*Ds*rmu;
   b1 =b+ADxsc;
   b2 =b-A*Dxsc;
   ADA=ADxs*A';
   ADA\[rry b1];
   dy=ans(:,1);
   dy1=ans(:,2);
   dd = b2'*dy1+kappa/tau+c'*Dxsc;
   if abs(dd) < toler^3,
       dtau=0; %keep tau unchanged
   else
   dtau=(rrt-b2'*dy)/dd;
   end;
   dy = dy+dtau*dy1;
   %
   % compute the rest directions
   ds=rd-A'*dy+dtau*c;
   dx=-Ds*(rmu+x.*ds);
   dkappa=r0+b'*dy-c'*dx;
%
%  choose step-size
%
   nora = min([dx./x;ds./s;dtau/tau;dkappa/kappa]);
   if nora < 0,
      nora = min(1,abs(alpha/nora));
   else
      nora=1;
   end;
%
% Update iterates
%
   y = y + nora*dy;
   s = s + nora*ds;
   x = x + nora*dx;
   tau = tau + nora*dtau;
   kappa=kappa+nora*dkappa;
%
% Duality gap
%
   mu = gamma*mu;
 end;
 % Check feasible or infeasible case and output final solutions
 if tau>kappa,
     x=x/tau;
     y=y/tau;
     s=s/tau;
 else
     x=x/kappa;
     y=y/kappa;
     s=s/kappa;
 end;
 time=toc
%