% Newton's method for solving the HSDSDP system after 
% dual-substitution to replace X from center equations X=mu*S^{-1}
%
%    mu A*S^{-1}  -b*tau     = mu(A*I-b)
%   -A'*y      +C*tau -S     = mu(C-I)
%   b'*y -mu C*S^{-1}-mu/tau = mu(-C*I-1)
%  
%  Assume: each of primal constraint matrices is
%          rank-one A_i=a_ia_i^T, i=1,...,m, 
%          where vector a_i is dimension n
%
%  Input Data
%      A: m x n matrix where A'  =[a_1, a_2, ..., a_m]
%      b: objective column vector in dimesnsion m
%      C: n x n objective symmetric matrix for the primal
%
% Output
%     X>=0   : approximate primal solution or infeasibility
%              certificate
%     S>=0   : dual slack variables
%              X*S =< toler.
%     y      : multipliers of equality constraints or 
%              infeasibility certificate
%     iter   : number of iterations
%
% Path-folloing strategy; details can be found in Sect. 
% 6.6 of L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [X,y,S,iter] = DualHOsdptaupath(A,b,C);
% Set parameters
%
 [m,n] = size(A);
 toler=1.e-6; 
 gamma=0.7;
%
% Set up initial solutions
%
 II= speye(n);
 S = II;
 tau=1;
 y = zeros(m,1);
 mu = 1;
%
% Set initial residuals
%
 rp0=sum(A'.^2)'-b;
 rd0=C-II;
 r00=-trace(C)-1;
 iter =0;
 tic;
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   % Reduce mu by a factor
   mu = gamma*mu;
   % Scale input data for applying Newton's method
   R    = chol(S);
   D    = inv(R);
   AD=A*D;
   M =AD*AD';
   DCD=D'*C*D;
   %
   % Set up Newton system initial residuals
   %
   rp= mu*diag(M)-tau*b -mu*rp0;
   rd= tau*C-A'*diag(y)*A-S -mu*rd0;
   r0= b'*y-mu*trace(DCD)-mu/tau -mu*r00;
   %
   % Start Newton step
   %
   % Construct linear system for directions dy and dtau
   ADDC=sum(AD'.*(DCD*AD'))';
   DrdD=D'*rd*D;
   %
   % Right-hand-side of the system 
   rry=-rp+mu*sum(AD'.*(DrdD*AD'))';
   rrt=-r0-mu*sum(DCD.*DrdD,'all');
   b1 =b+mu*ADDC;
   b2 =b-mu*ADDC;
   % Solve for dy and dtau separately 
   (mu*M.^2)\[rry b1];
   dy=ans(:,1);
   dy1=ans(:,2);
   dd=b2'*dy1+mu*(1/tau^2+sum(DCD.*DCD,'all'));
   if abs(dd) < toler^2,
       dtau=0;
   else
       dtau=(rrt-b2'*dy)/dd;
   end;
   % Compute dy first
   dy = dy+dtau*dy1;
   % Compute (scalled) dS
   DdSD=DrdD-AD'*diag(dy)*AD+dtau*DCD;
%
% Update Newton iterates
%
   y = y + dy;
   S = R'*(II + DdSD)*R;
   tau = tau + dtau;
%
 end;
 % Recover the primal solution if needed
 R    = chol(S);
 D    = inv(R);
 AD=A*D;
 M =AD*AD';
 yy=(M.^2)\(b*tau-mu*diag(M));
 X=D*(mu*II+AD'*diag(yy)*AD)*D';
 kappa=mu/tau;
 % Check feasible or infeasible case and output final solutions
 if tau>kappa,
     X=X/tau;
     y=y/tau;
     S=S/tau;
 else
     X=X/kappa;
     y=y/kappa;
     S=S/kappa;
     disp('The problem is either infeasible or unbounded.');
 end;
 time=toc
%