%======================================================
%  Matlab Demonstration of SDP Relaxation for Sensor Network 
%  Localization followed by the steepest-descent and the 
%  dimension-reduced 2nd-order trust-region methods
%  (Lecture Note 12, Slide 26)
%
%  Need to call public SDP solver SEDUMI
%
%  Input:
%     PP: sensor point positions
%     m: number of assumed known points (anchors), they are PP(:,1:m)
%     n: number of total points, they are PP(:,1:n)
%     Radius: like a radio range; the distance estimation of two points 
%         whose distance is less than Radius will be used by the algorithm. 
%     nf: noise factor - the distance estimation between two nearby points
%         is computed by true-distance^2*(1+randn(1)*nf)
%     degree: maximum number of edge distances on a node used in algorithm
%
%  Output: centers positions of n spheres.
%  Output:
%     X_opt: position estimation for all n-m unknown points computed by 
%            the algorithm/model
%     Y_opt: the output SDP matrix representing the second moment 
%            approximation of X_opt. If Y_opt=X_opt'*X_opt, the extimation
%            is exact.
%     tvar(i): Y_opt(i,i)-X_opt(:,i)'*X_opt(:,i), ith point's trace
%              "variance"; the smaller the trace, the more accurate 
%              localization of the point.
%     error(i): norm(X_opt(:,i)- PP(:,m+i)) - distance between the true 
%               position and computed position from by the algorithm/model.
%     figure 1: SDP solution accuracty 
%     figure 2: Solution accuracry after the steepest gradient refinement
%               blue diamond: anchor point
%               green circle: true sensor points
%               red *: computed sensor points
%
%  Details can be found in HW6.8 and Sect. 6.2 of
%  L&Y, Linear and nonlinear programming, 5th edition
%======================================================%  
function [Y_opt,X_opt]=SDPSDMDRTRsnlsedumi(PP,m,n,Radius,nf,degree)
%
    close all;
    PP=[PP(:,m+1:n) PP(:,1:m)];
    dd = n-m;
    Anchor = PP(:,dd+1:n);
    %
    e = [ones(dd,1); sum(PP(:,dd+1:n),2)]/sqrt(n);
    C=1.4*nf*vec(e*e'-speye(dd+2,dd+2));
    %
    b=zeros(3,1);
    a=sparse([zeros(1,dd) 1 0]');
    A=vec(a*a');
    b(1)=1;
    a=sparse([zeros(1,dd) 0 1]');
    A=[A vec(a*a')];
    b(2)=1;
    a=sparse([zeros(1,dd) 1 1]');
    A=[A vec(a*a')];
    b(3)=2;
    %
    % Compute pair-wise distances within R limit between known to unknown 
    % sensor-point and uunknown to unkown within the radio range and set up
    % the constraint matrix.
    %
    dij=sparse(zeros(n,n));
    mtotal=3;
    for i=1:dd
        flag = 0 ;
        for j=(i+1):n
            %
            rr= norm(PP(:,i)-PP(:,j));
            %
            if rr < Radius && flag < degree
               flag =flag+1;
               mtotal=mtotal+1;
               dij(i,j)=rr*sqrt(max(0,(1+randn(1)*nf)));%add noise
               if j <= dd
                  % jth is an unknown point (sensor)
                  a=sparse([zeros(1,i-1) 1 zeros(1,j-i-1) -1 zeros(1,dd-j+2)]');
                  A(:,mtotal)=vec(a*a');    
               else
                  % jth is a known point (anchor)
                  a=sparse([zeros(1,i-1) 1 zeros(1,dd-i) -PP(:,j)']');
                  A(:,mtotal)=vec(a*a');
               end;
               b(mtotal) = (dij(i,j))^2; 
            end;
        end;
    end;
    % 
    % Setup to call SEDUMI
    %
    mequal=mtotal
    K.s=[dd+2];
    K.s=[K.s ones(1,mtotal-3)];
    pars.eps=1.e-3;
    tic
    A=[A; sparse([zeros(mtotal-3, 3) -eye(mtotal-3)])];
    C=[C;ones(mtotal-3,1)];
    pars.stepdif=0;
    [x,y,info] = sedumi(A,b,C,K,pars);
    Y_opt=mat(x(1:(dd+2)^2));
	X_opt=Y_opt(dd+1:dd+2,1:dd);
    Y_opt=Y_opt(1:dd,1:dd);
    tvar=0;
    sdpt=toc
    % Plot the position estimations produced by the SDP solver
    figure(1);
    hold on ;
    plot(PP(1,dd+1:n),PP(2,dd+1:n),'d');       
    plot(X_opt(1,:),X_opt(2,:),'*r');
    plot(PP(1,1:dd),PP(2,1:dd),'og');
    for i=1:dd
        plot([X_opt(1,i)  PP(1,i)] , [X_opt(2,i)  PP(2,i)]);
    end;
    hold off
    % Compute the variance/trace and true error
    for j=1:dd
    tvar(j)=max(0,Y_opt(j,j)-(X_opt(:,j)'*X_opt(:,j)));
    error(j)=norm(X_opt(:,j)-PP(:,j));
    end;
    %
    % Apply the steepest descent method (SDM) for nonlinear least-squares 
    % minimization to refine the positions
    %
    tic
    [Xiter,objhist] = steepdescent2(Anchor,X_opt,m,dij);
    %
    t=toc
    figure(2)
    %
    hold on;
    plot(Anchor(1,:),Anchor(2,:),'d');
    hold on; 
    plot(PP(1,1:dd),PP(2,1:dd),'og');
    hold on;
    plot(Xiter(1,:),Xiter(2,:),'*r');
    %          
    for i = 1:dd
        hold on ;
        line([Xiter(1,i)  PP(1,i)] , [Xiter(2,i)  PP(2,i)]);
    end
    hold off
    % 
    % Apply the dimension-reduced 2nd-order trust-region method (DRTR) for 
    % nonlinear least-squares minimization to refine the positions
    %
    tic
    [Xiter,objhist] = DRTRNewtonSNL2(Anchor,X_opt,m,dij);
    %
    t=toc
    figure(3)
    %
    hold on;
    plot(Anchor(1,:),Anchor(2,:),'d');
    hold on; 
    plot(PP(1,1:dd),PP(2,1:dd),'og');
    hold on;
    plot(Xiter(1,:),Xiter(2,:),'*r');
    %          
    for i = 1:dd
        hold on ;
        line([Xiter(1,i)  PP(1,i)] , [Xiter(2,i)  PP(2,i)]);
    end
    hold off
%%*************************************************************************
%%*************************************************************************
%% steepdescent: steepest descent with back-tracking line search to 
%%               minimize 
%%
%% f(X) = sum_{j<=k} (norm(xj-xk)+djk)^2 + sum_{j,k} (norm(xj-ak)-djk)^2
%%
%% input: X0 = [sensor position, anchor]
%%        DD = [sensor-sensor distance, senor-anchor distance
%%              anchor-sensor distance,  0]
%%
%% child functions: gradfun2, objfun2
%% 4 April 2007
%%*************************************************************************

  function [Xiter,objhist] = steepdescent2(anchor,X0,m,DD,maxit)

  if (nargin < 5); maxit = 2000; end
 
  [dummy,n] = size(X0);
  [dummy,m] = size(anchor); 
  if (size(DD,2) ~= n+m) 
     error('steepdescent: dimension of X0 or DD not correct')
  end
  D1 = DD(1:n,1:n); 
  [II,JJ,dd] = find(triu(D1)); 
  ne = length(II); 
  S  = sparse(II,1:ne,1,n+m,ne)-sparse(JJ,1:ne,1,n+m,ne);   
  dd = dd'; 
  if (m > 0)
     D2 = DD(1:n,n+[1:m]); 
     [I2,J2,d2] = find(D2); 
     ne2 = length(I2);
     S2 = sparse(I2,1:ne2,1,n+m,ne2)-sparse(n+J2,1:ne2,1,n+m,ne2);
     S  = [S,S2]; 
     dd = [dd, d2']; 
  end
  Sg = S'; Sg = [Sg(1:length(dd),1:n), sparse(length(dd),m)]; 
  X0 = [X0, anchor]; 
  obj  = objfun2(X0,S,dd); 
  gradx = gradfun2(X0,S,dd,Sg);
  objhist.obj(1) = obj; 
  objhist.grad(1) = sqrt(max(sum(gradx.*gradx)));
  Xiter = X0;
%%
  for iter = 1:maxit
     objnew = inf;
     objold = objhist.obj(iter); 
     gradx = gradfun2(Xiter,S,dd,Sg);
     alpha = 0.2; count = 0;
     while (objnew > objold) & (count < 20)
        alpha = 0.5*alpha; count = count + 1; 
        Xnew  = Xiter - alpha*gradx;
        objnew = objfun2(Xnew,S,dd);
     end 
     Xiter = Xnew;
     objhist.obj(iter+1)  = objnew;
     objhist.grad(iter+1) = sqrt(max(sum(gradx.*gradx)));
     objhist.step(iter+1) = alpha; 
     if (abs(objnew-objold)/(1+abs(objold)) < 1e-10); break; end 
  end
  Xiter = Xiter(:,1:n); 
%%*************************************************************************
%%*************************************************************************
%% DRNewton: dimention-reduced 2nd-order trust-region Newton to minimize 
%%
%% f(X) = sum_{j<=k} (norm(xj-xk)+djk)^2 + sum_{j,k} (norm(xj-ak)-djk)^2
%%
%% input: X0 = [sensor position, anchor]
%%        DD = [sensor-sensor distance, senor-anchor distance
%%              anchor-sensor distance,  0]
%%
%% child functions: gradfun2, objfun2
%% 5 March 2022
%%*************************************************************************

  function [Xiter,objhist] = DRTRNewtonSNL2(anchor,X0,m,DD,maxit)

  if (nargin < 5); maxit = 200; end
 
  [dummy,n] = size(X0);
  [dummy,m] = size(anchor); 
  if (size(DD,2) ~= n+m) 
     error('DRTRNewton: dimension of X0 or DD not correct')
  end
  % prepare matrix to symbolically build the gradient of objective
  D1 = DD(1:n,1:n); 
  [II,JJ,dd] = find(triu(D1)); 
  ne = length(II); 
  S  = sparse(II,1:ne,1,n+m,ne)-sparse(JJ,1:ne,1,n+m,ne);   
  dd = dd'; 
  if (m > 0)
     D2 = DD(1:n,n+[1:m]); 
     [I2,J2,d2] = find(D2); 
     ne2 = length(I2); %the total number of distance terms
     S2 = sparse(I2,1:ne2,1,n+m,ne2)-sparse(n+J2,1:ne2,1,n+m,ne2);
     S  = [S,S2]; 
     dd = [dd, d2']; 
  end
  Sg = S'; Sg = [Sg(1:length(dd),1:n), sparse(length(dd),m)]; 
  X0 = [X0, anchor]; 
  obj  = objfun2(X0,S,dd);
  gradx0 = gradfun2(X0,S,dd,Sg); %compute the gradient
  objhist = obj;
  Dx=-gradx0/10; %take one SDM step
  Xiter = X0+Dx;
  lambda=10; %set the initial trust-region regularization weight
%%
  for iter = 1:maxit
     objnew = inf;
     objold = objhist(iter);
     % compute the new gradient
     gradx = gradfun2(Xiter,S,dd,Sg);
     %
     % approximate reduced Hessian*gradient at xnew
     Hg = 10*(gradfun2(Xiter+gradx/10,S,dd,Sg)-gradx);
     % approxinate Hessian*(xnew - xold)
     HDx= gradx - gradx0;
     % compute the approximate 2-dimensional reduced gradient&Hessian
     b1=sum(sum(gradx.*gradx));
     b2=sum(sum(gradx.*Dx));
     Q12=sum(sum(Hg.*Dx));
     QK=[sum(sum(gradx.*Hg)) -Q12;-Q12 sum(sum(Dx.*HDx))];
     %
     %BK=[b1 -b2;-b2 sum(sum(Dx.*Dx))];
     %emin=min(eig(QK, BK))
     %
     emin=min(eig(QK)); %check the definiteness of the reduced Hessian
     if (emin > 0), %if convex, take Newton step and do back-tracking
         count=0;
         objnew=inf;
         while (objnew > objold) & (count < 20)
           (QK+(1.e-6+0.01*lambda)*eye(2))\[b1;-b2];
           alphag=ans(1);alpham=ans(2);
           Dx=-alphag*gradx+alpham*Dx; %Reduced Newton direction 
           %count=0; alpha = 1; 
           % back-tracking
           %while (objnew > objold) & (count < 20)
           %Dx=alpha*Dx;
           Xnew = Xiter+Dx;
           objnew = objfun2(Xnew,S,dd);
           %alpha = 0.5*alpha; 
           lambda = 1.5*lambda; %reduce the trust-region radius
           count = count + 1;
         end;
     else %if noncovex do trust-region Newton step and back-tracking
         count=0;
         objnew=inf;
         while (objnew > objold) & (count < 20)
             % Take trust region step
             (QK+(abs(emin)+lambda)*eye(2))\[b1;-b2];
             alphag=ans(1);alpham=ans(2);
             Dx=-alphag*gradx+alpham*Dx;
             Xnew = Xiter+Dx;
             objnew = objfun2(Xnew,S,dd);
             lambda = 1.5*lambda; %reduce the trust-region radius
             count = count + 1; 
         end;
     end;
     Xiter = Xnew;
     gradx0=gradx;
     objnew=objfun2(Xiter,S,dd);
     objhist  = [objhist objnew];
     lambda=max(1.e-8,lambda/2); %decrease the regularizaon weight
     if (abs(objnew-objold)/(1+abs(objold))<1e-10);break;end 
  end
  Xiter = Xiter(:,1:n); 
%%*************************************************************************
%% Find the function value
%% f(X) = sum_{j<=k} (norm(xj-xk)-djk)^2 + sum_{j,k} (norm(xj-ak)-djk)^2
%%
%% input: X = [sensor position, anchor]
%%*************************************************************************

  function  objval = objfun2(X,S,dd);

  Xij = X*S; 
  %% Edited by Zhisu.%%%%%%%
  if size(dd,1)>1 
      dd=dd';
  end
  %%%%%%%%%%%%%%%%%%%%%%%%%%
  tmp = sqrt(sum(Xij.*Xij))-dd; 
  objval = norm(tmp)^2;

%%*************************************************************************
%% Find the gradient of the function 
%% f(X) = sum_{j<=k} (norm(xj-xk)-djk)^2 + sum_{j,k} (norm(xj-ak)-djk)^2
%%
%% input: X = [sensor position, anchor]
%%*************************************************************************

  function  G = gradfun2(X,S,dd,Sg);

  ne = length(dd); 
  Xij = X*S; 
  %% Edited by Zhisu.%%%%%%%
  if size(dd,1)>1 
      dd=dd';
  end
  %%%%%%%%%%%%%%%%%%%%%%%%%%
  %tmp = 1-dd./(eps+sqrt(sum(Xij.*Xij))); 
  tmp = 1-dd./(max(eps, sqrt(sum(Xij.*Xij)))); 
  G = Xij*spdiags(2*tmp',0,ne,ne); 
  G = G*Sg;
%%*************************************************************************
