%======================================================
%  Matlab implementation of the
%  Dual potential reduction algorithm for solving SDP
%  
%      maximize    b'*y
%      subject to  C-A'*y is psd, 
%
%  Assume: both primal and dual have interior feasible solutions;
%          (rank-one) A_i=a_ia_i^T in primal form, i=1,...,m, 
%          where vector a_i is dimension n
%
%  Input Data
%      A: m x n matrix where A'  =[a_1, a_2, ..., a_m]
%      b: objective column vector in dimesnsion m
%      C: nxn objective symmetric matrix for the primal
%      y: any dual interior feasible point
%      z: an upper bound on optimal objective value
%      n: dimmension of C (optional)
%     rho: potential function parameter (>= n+sqrt(n)) (optional)
%
%  Output
%      X: primal solution
%      (y,S): dual solution
%   Details can be found in Sect. 6.6 of
%   L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [X,y,S] = dsdpMLB(A,b,C,y,z);
%
%  Set initial algorithm parameters
%
if exist('toler') ~= 1 
   toler=1.e-3; 
end
if exist('alpha') ~= 1 
   alpha=0.75;    
end
if exist('n') ~= 1 
   [m,n]=size(A);    
end 
if exist('rho') ~= 1 
   rho=2*n;   
end 
%
[m,n]=size(A);
gap=1;
iter=0;
while gap > toler,
iter=iter+1
gap     = z - b'*y;
S       = C-A'*diag(y)*A;
potenf  = rho*log(gap)-log(det(S));
R       = chol(S);
R       = inv(R);
B       = A*R;
M       = B*B';
gb      = diag(M);
(M.*M+1.e-6*eye(m))\[b gb];
dy1     = ans(:,1); dy2=ans(:,2);
dy      = (rho/gap)*dy1-dy2;
%Pk      = B'*diag(dy)*B;
normpk  = sqrt(dy'*((rho/gap)*b-gb));
xupdate =0;
while normpk < alpha, 
  xupdate=1;
%  Xk     = (gap/rho)*(R*(eye(n)+B'*diag(dy)*B)*R');
  gap    = (gap/rho)*(n+dy'*gb);
  z      = gap + b'*y;
  dy     = (rho/gap)*dy1-dy2;
%  Pk     = B'*diag(dy)*B;
  normpk = sqrt(dy'*((rho/gap)*b-gb));
end;
%
% Update the primal
%
if xupdate ==1,
X     = (gap/rho)*(R*(eye(n)+B'*diag(dy)*B)*R');
end;
%
%  Update the dual solution
%
y     = y+(alpha/normpk)*dy;
end;
%


