% The matlab function is to minimize sum of fractional objective
% (such as sum of sharpe-ratios)
% \sum_{i} ( y(i) - X(i,:)*beta )^2/( Z(i,:)*alpha )
%
%  Need to call public SDP solver DSDP (64 bits)
%
% input data:
%       X(n x d) general matrix
%       Y(n x 1) observation vector
%       Z(n x p) non-negative matrix
%       n: number of terms
%       d: dimension of beta
%       p: dimension of alpha
%   output optimal vectors:
%       alpha(p x 1): non-negative vector and its sum equal to 1
%       beta(d x 1): general vector
%   Supplemental unconstrained optimization example for Chap. 7 
%   L&Y, Linear and nonlinear programming, 5th edition
%
% In Matlab platform, just type [alpha,beta]=Regretmin(n,d,p,X,Y,Z); 
% and enter, it needs DSDP to run
% ======================================================% 
function [alpha,beta]=Regretmin(n,d,p,X,Y,Z)
%
% set up 2x2 sdp matrix cone 
for i=1:n
    % inpu 2x2 matrix coefficients of alpha
    AACi=sparse(0,0);
    AACi = [ AACi [sparse(2,p); -sparse(Z(i,:))] ];
    % input 2x2matrix coefficients of beta
    AACi = [ AACi [sparse(1,d); sparse(X(i,:)); sparse(1,d)] ];
    % input 2x2 matrix coefficients of gamma(i)
    temp = sparse(3,n);
    temp(1,i) = -1;
    AACi = [ AACi temp ];
    % input 2x2 matrix of the ith right-hand-side of the dual
    AACi=[ AACi sparse([ 0 Y(i) 0 ]') ];
    AC{i,1} = 'SDP';
    AC{i,2} = [2];
    AC{i,3} = AACi;
end
% set up nonnegative cone for alpha
AC{n+1,1}='LP';
AC{n+1,2}=p+1;
AC{n+1,3}=[[sparse(ones(1,p));-speye(p)] sparse(p+1,d) sparse(p+1,n) sparse([1 sparse(1,p)]')];
% set up the dual objective vector; note that SDP dual variable vector y=[alpha; beta; gamma]
b=[zeros(1,p) zeros(1,d) -ones(1,n)]';
% call dsdp solver
[STAT,y,XX]=dsdp64(b,AC);
% output alpha and beta
alpha=y(1:p);
beta =y(p+1:p+d);
% end of the function