%======================================================
%  Matlab implementation of the cyclic block-coordinate
%  gradient/descent method for linear least-squares
%  
%      minimize    \|Ax-b\|^2
%
%  Input Data
%      A: m x n matrix
%      b: vector
%      nb: number of blocks such that n divided
%          by nb is an integer)
%  Output
%      x: latest iterative solution
%
%   Details can be found in Sect. 8.8 of
%   L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
%
function [x]=CyclicBCDglls(A,b,nb,maxiter,toler)
if exist('maxiter') ~= 1 
   maxiter=500; 
end
if exist('toler') ~= 1 
   toler=1.e-3; 
end
beta=max(eigs(A'*A))/nb
tic
[m,n]=size(A);
x=zeros(n,1);
nv=floor(n/nb);
% set initial residual vector
Axb=A*x-b;
%
iter=0;
for k=1:maxiter,
  % Update x following the cyclic order 1 to nb
  for ii=1:nb,
      % find the variable indexes of the (ii)th block
      il=(ii-1)*nv+1;
      iu=il+nv-1;
      % construct sub-A matrix according to the block
      Ap=A(:,il:iu);
      % find the corresponding gradient and update the block
      dd=-(1/beta)*(Ap'*Axb);
      x(il:iu)=x(il:iu)+dd;
      % updat the residual
      Axb=Axb+Ap*dd;
      % go to the next block
  end
  iter=iter+1;
  % Check stopping criteria
  if norm(A'*Axb)<toler, break, end;
end;
toc
iter=iter
%  