%======================================================
%  Matlab implementation of the double-sweep block-
%  coordinate method for linear least-squares
%  
%      minimize    \|Ax-b\|^2
%
%  Input Data
%      A: m x n matrix
%      b: vector
%      nb: number of blocks such that n divided
%          by nb is an integer)
%  Output
%      x: latest iterative solution
%
%   Details can be found in Sect. 8.8 of
%   L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
%
function [x]=DsweepBCDlls(A,b,nb,maxiter,toler)
if exist('maxiter') ~= 1 
   maxiter=100; 
end
if exist('toler') ~= 1 
   toler=1.e-3; 
end
tic
[m,n]=size(A);
x=zeros(n,1);
nv=floor(n/nb);
% set initial residual vector
Axb=A*x-b;
%
iter=0;
for k=1:maxiter,
  % Update x following the cyclic order
  for ii=1:nb,
      il=(ii-1)*nv+1;
      iu=il+nv-1;
      % find the nv indexes of the (ii)th block according to or
      % construct sub-A matrix according to the block
      Ap=A(:,il:iu);
      % substract them from the residual 
      Axb=Axb-Ap*x(il:iu);
      % update the block
      x(il:iu)=-Ap\Axb;
      % add updated block back to the residual
      Axb=Axb+Ap*x(il:iu);
      % go to the next block
  end
  for ii=1:(nb-1),
      il=(nb-ii-1)*nv+1;
      iu=il+nv-1;
      % find the variable indexes of the (ii)th block
      % substract them from the current residual 
      Axb=Axb-A(:,il:iu)*x(il:iu);
      % construct sub-A matrix according to the block
      Ap=A(:,il:iu);
      % update the block
      x(il:iu)=-Ap\Axb;
      % add updated block back to the residual
      Axb=Axb+A(:,il:iu)*x(il:iu);
      % go to the next block
  end
  iter=iter+1;
  % Check stopping criteria
  if norm(A'*Axb)<toler, break, end;
end;
toc
iter=iter
%   