% This program solves the linearly constrained non-convex
% program 
%
% min   0.5x^TQx+c^Tx+mu(\sum x_j^{.5})
% s.t.  e^Tx = 1, x >= 0.
%
% via sequential 2nd-order quadratic programming or
% interior trust-region
%
%  Input 
%      Q: covariance matrix
%      c: expectation vector
%      mu: regularization weight factor
%     
%  Output
%     x>=0  : feasible solution: Ax = b,
%     y     : Lagrange multiplier
%  
%  Output
%     x>=0  : (sparse) portfolio solution e^Tx=1 
%  
%  Example 1 in Sect. 11.6 and Algorithm in Sect. 12.7
%  (with full Hessian including the concave part)
%  of L&Y, Linear and nonlinear programming, 5th edition
%======================================================%
function [x,y,iter]=Halfnormport2ndH(Q,c,mu,toler,alpha)
if exist('mu') ~= 1
  mu = 0.1;
end;
if exist('toler') ~= 1
  toler = 1.e-4;
end;
if exist('alpha') ~= 1
  alpha = 0.5;
end;
[m,n]=size(Q);
A=ones(1,n);
x  = ones(n,1)/n;
gap=1;
lam=1;
iter=0;
%
%  Repeatly solving the interior-trust region subproblem
%
 while (gap >= toler) & (iter < 101),
   % generate affine-scaling direction
   gg = -x.*(Q*x+c)-(mu/2)*sqrt(x);
   % construct affine-scaling Hessian
   DD = sparse(1:n,1:n,x,n,n);
   AD = A*DD;
   DHD = DD*Q*DD-(mu/4)*diag(sqrt(x));
   % set up the trust-region size
   lam=max([lam toler^2]);
   y = [DHD+lam*speye(n) AD';AD 0]\[gg;0];
   dd = y(1:n);
   y = -y(n+1);
   % compute the residual of the projected gradient
   % and adjust trust-rgion size for the next step 
   gap = norm(dd);
   if gap < 1,
     lam=lam/2;
   else
     lam=1.5*lam;
   end;
   % compute step size to stay in the interior
   nora = min(dd);
   nora = abs(alpha/nora);
   %
   % Update iterates
   %
   x = x + nora*(x.*dd);
   iter=iter+1;
 end;
 % end