%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Full implementation of the interior-trust region method
% for solving general linear-constrained nonlinear programs:
%
%      min   f(x,z)
%      s.t.  Ax = b, x >= 0, z free.
%
%  Input 
%      x0: interior (positive) primal feasile solution with dimension n
%      z0: initial primal solution of z with dimension p
%      y0: interior Lagrange multipliers, i.e, nabla_x f(x0,z0)-A'*y0 > 0
%      A: Sparse constraint matrix.
%      b: Right-hand column vector
%      toler: relative stopping tolerance: the objective value close to 
%             the optimal one in the range of tolerance. 
%             Default value: 1.e-6.
%      alpha: step size: alpha > 0. Default value: 1.
%      gamma: weight parameter: 0< gamma <=1. Default value: .7.
%
%  Matlab functions:
%      gradien(x,z): return the gradient vector of the objec. function at 
%      x and z
%      hessian(x,z): return the hessian matrix of the objec. function at 
%      x and z.
%     
%  Output
%     x>=0  : primal nonnegative solution: Ax = b,
%     z     : primal free variable solution
%     y,s>=0: Lagrange (dual) solution: s = nabla f(x) - A^Ty,
%             x^Ts/n =< toler.
%  
%  Guide: type ITRsollcnp and return.
%
%   Details in Sect. 12.7 of
%   L&Y, Linear and nonlinear programming, 5th edition
%======================================================%
%
% Set parameters
%
 [p,m] = size(z0);
 [m,n] = size(A);
 if exist('toler') ~= 1 
   toler=1.e-6; 
 end
 if exist('gamma') ~= 1 
   gamma=.6;    
 end
 if exist('alpha') ~= 1 
   alpha=0.3;    
 end
%
% Set initial points
%
 x=x0;
 z=z0;
 y=y0;
 g=gradienf(x,z,c);
 s=g(1:n)-A'*y;
 mu = x'*s/n;
 iter =0; 
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   %
   mu = gamma*mu;
   % Form the right-hand-vector
   xr = ones(n,1)./x;
   qqq = [mu*xr-s;-g(n+1:n+p);-(A*x-b)];
   %
   % Get the objective hessian matrix
   %
   MM = hessianf(x,z,c);
   %
   % Add barrier hessian
   %
   XX=sparse(1:n,1:n,xr.^2);
   MM(1:n,1:n)=MM(1:n,1:n)+mu*XX;
   %
   % Sove the system of linear equartions
   %
   dx=[MM [-A';sparse(p,m)];A sparse(m,p+m)]\qqq;
   %
   % Split
   %
   dy=dx(n+p+1:n+p+m);
   dz=dx(n+1:n+p);
   dx=dx(1:n);
   %
   % Update
   %
   beta = 1;
   ss = 0;
   while ss <= 0,
    xx = x + beta*dx;
    yy = y + beta*dy;
    zz = z + beta*dz;
    g=gradienf(xx,zz,c);
    s = g(1:n) - A'*yy;
    beta = alpha*beta;
    ss = min([xx;s]);
   end
   x=xx;
   y=yy;
   z=zz;
   mu = x'*s/n;
 end
 %
  iter
 return



