%======================================================
%  Matlab implementation of the (interior) trust-region
%  method for indefinite box-constrained quadratic 
%  miniization with a small barrier
%  
%      minimize 0.5x'Qx+c'x +(barrier) s.t. 0<= x <=1
%
%  Input 
%      Q: Sparse symmetric objective matrix
%      c: objective column vector
%      toler: relative stopping tolerance: the objective value close to 
%             the local optimal one in the range of tolerance. 
%             Default value: 1.e-5.
%      beta : step size: 0 < beta < 1. Default value: .8.
%
%  Output
%     x: (local) optimal solution
%     y: optimal dual solution (Lagrangien multiplier) for x<=1
%     obhis : objective value history vs iterations
%
%   Algorithm details can be found in Sect. 12.7 and and 13.4 of
%   L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [x,y,obhis]=ITRBarrierboxqp(Q,c,toler,beta)
%
 if exist('toler') ~= 1 
   toler=1.e-5; 
 end
 if exist('beta') ~= 1 
   beta=0.8;    
 end
 alpha = 0.9;
 [m,n] = size(Q);
% 
% set the initial solution
%
 x1=0.5*ones(n,1);
 x1 = x1 + 0.01*rand(n,1);
 x2=1-x1;
 x=[x1;x2];
 %
 disp('Search for an optimal solution:');
 %
 obvalue=x1'*(Q*x1)/2+c'*x1;
 obhis=[obvalue];
 lower =-inf;
 zhis=[lower];
 gap=1;
 %
 % set trust-region parameter
 %
 lamda=max([1 abs(obvalue)/sqrt(sqrt(2*n))]);
 %lamda=max([n abs(obvalue)/sqrt(sqrt(n))]);
 %
 iter=0;
 %
 while gap >= toler,
    iter=iter+1; 
% reduce trust-region parameter
% lamda=(1.-beta/max([1,n^(1/12)]))*lamda;
% 
    lamda=(1.-beta)*lamda;
% if gap <= 5*toler;
%   lamda = lamda/2;
% end;
    go=0;
    gg = Q*x1+c-toler./min(x1,1-x1);% add a little barrier term
    X1 = spdiags(x1,0,n,n);
    X1 = X1*Q*X1;
    X12= spdiags(x1./x2,0,n,n);
%    x12 = x1./x2;
%
%  Repeatly solve an ellipsoid constrained QP problem by solving a linear
%  system equation until find a positive solution.
%
    while go <= 0,
%
        u=(X1+lamda*speye(n,n)+(lamda*X12)*X12)\(-x1.*gg);
        u2=-(x1.*u)./x2;
%        u2=(x2.*y)/lamda;
        y = (lamda*u2)./x2;
        u=[u;u2];
% 
        xx=x+x.*u;
        xx1=xx(1:n);
        go=min(xx);
        if go > 0,
            ob=xx1'*(Q*xx1)/2+c'*xx1;
            go = min([go obvalue-ob+eps]);
        end;
        lamda=2*lamda;
        if lamda >= (1+abs(obvalue))/toler,
            disp('The problem seems unbounded.');
            x=xx1;
            return
        end; 
    end;
%
% set the step-size
%   
    nora = min(u);
    if nora <= 0,
        nora=alpha/max(-nora,1);
 %elseif nora == 0,
 %  nora=alpha;
    else
        nora=inf;
    end
%
    u =  x.*u;
    u1=u(1:n);
    w1 = u1'*(Q*u1);
    w2 = -u1'*gg;
    if w1 > 0,
        nora=min([w2/w1,nora]);
    end;
    if nora == inf,
        ob = -inf;
        gap = 0;
        disp('The problem is unbounded.');
        return
    else
        x =x+nora*u;
        x1=x(1:n);
        x2=x(n+1:2*n);
        ob=x1'*(Q*x1)/2+c'*x1;
    end;
%
%
    obhis=[obhis ob]; 
    comp=[Q*x1+c-y;-y]; 
    if min(comp)>=0, 
        zhis(iter+1)=ob-x'*comp;
        lower=zhis(iter+1),
        gap=(ob-lower)/(1+abs(ob));
        obvalue=ob;
    else
        zhis(iter+1)=zhis(iter);     
        lower=zhis(iter+1);
        gap=(obvalue-ob)/(1+abs(ob));
        obvalue=ob;     
    end;
 end;
 disp('A (local) optimal solution is found.');
 x=x1;
 y=full(y);
% iter=iter




