% Matlab implementation of the second-order interior-trust-region
% method by dynamically reducing barrier weight for solving
% linearly constrained (convex) program
%
%      min   f(x)
%      s.t.  Ax = b (y), x >= 0.
%
%  Input 
%      A: Sparse constraint matrix.
%      b: right-hand-side vector
%      c: parameter of objective function
%      x0: interior (positive) primal feasile solution with dimension n
%      toler: relative stopping tolerance: the objective value close to 
%             the optimal one in the range of tolerance. 
%             Default value: 1.e-6.
%      alpha: inverse radias, Default value: 0.5
%      gamma: barrier weight reduction rate, 0<gamma<=1, Default value: .5
%
%  Matlab functions:
%  objgradien(x,c): return the gradient vector of the objective 
%  objhessian(x,c): return the hessian matrix of the objective at 
%     
%  Output
%     x>=0  : primal nonnegative solution: Ax = b,
%     y,s   : Lagrange (dual) solution, s = nabla f(x) - A^Ty,
%             x^Ts/n =< toler.
% 
% Algorithm details can be found in Sect. 13.4 
% of L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [x,y,s,iter] = ITRlccp(A,b,c,x0,toler,alpha,gamma);
% Set parameters
%
 [m,n] = size(A);
 if exist('toler') ~= 1 
   toler=1.e-6; 
 end
 if exist('alpha') ~= 1 
   alpha=0.5;    
 end
 if exist('gamma') ~= 1 
   gamma=.5;    
 end
%
% Input initial solutions, and set initial dual and barrier weight 
%
 x=x0;
 g=objgradien(x,c);
 mu = norm(g);
 iter =0;
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   % Scaling constraints by X
   XX=sparse(1:n,1:n,x);
   AX=A*XX;
   % Form the right-hand-vector of the scaled KKT system
   % qqq = [mu*ones(n,1)-x.*g+AX'*y;zeros(m,1)];
   qqq = [mu*ones(n,1)-x.*g;zeros(m,1)];
   %
   % Get the objective hessian matrix and scale it by X
   %
   H = objhessian(x,c);
   H =XX*H*XX;
   %
   % Solve the system of linear equartions
   %
   go=0;
   lambda=alpha*mu;
   % adjust the trust-region radius
   while go <= 0,
     lambda=lambda/alpha;
     dx=[H+lambda*speye(n) AX';AX sparse(m,m)]\qqq;
     % Construct primal and dual directions
     y=-dx(n+1:n+m);
     dx=dx(1:n);
     x=x+x.*dx;
     %y=y+dy;
     go = min(x);
   end
   g=objgradien(x,c);
   % reduce mu
   mu = gamma*mu;
 end;
% construct dual slack as by-product
 s = g - A'*y;
%
 return




