%======================================================
%  Matlab demonstration of solving general QP by the 
%  Multi-Block Cyclic ADMM 
%  (beta = 1, multi-block of x, last block: s):
%
%      minimize      0.5x'Qx + c'x
%      subject to     Ax     = b,  (y dimension m)
%                     x -  s = 0,  (z dimesnion n)
%                     s>=0
%
%      Input: A, Q, b, c, nb(number of blocks)
%             maxiter: the number of iterations
%      Output:primal x, dual y, and
%             the number of iteration before termination  
%
%  Algorithm details can be found in Sect. 14.6
%  L&Y, Linear and nonlinear programming, 5th edition
%======================================================%
function [x,y,iter]=MBCADMMQP(Q,A,b,c,nb,maxiter)
if exist('maxiter') ~= 1 
   maxiter=100; 
end
tic
toler=1.e-4;
[m,n]=size(A);
beta=1;
% prepare the block data
nv=floor(n/nb);
ATA=zeros(nv,n);
for ii=1:nb,
    il=(ii-1)*nv+1;
    iu=il+nv-1;
    AB=A(:,il:iu);
    ATA(:,il:iu)=inv(Q(il:iu,il:iu)+beta*(AB'*AB)+beta*eye(nv));
end;
clear AB;
%  set initial solution values
x=ones(n,1);
s=x;
y=zeros(m,1);
z=zeros(n,1);
%  compute initial gradients and constraint residual
Qxc=Q*x+c;
Ax=A*x;
iter=0;
for k=1:maxiter,
% Update x sequentially
  for ii=1:nb,
      il=(ii-1)*nv+1;
      iu=il+nv-1;
      Qxc=Qxc-Q(:,il:iu)*x(il:iu);
      Ax=Ax-A(:,il:iu)*x(il:iu);
      cc=Qxc(il:iu)-z(il:iu)-beta*s(il:iu)-A(:,il:iu)'*(y+beta*(b-Ax));
      x(il:iu)=-ATA(:,il:iu)*cc;
      Qxc=Qxc+Q(:,il:iu)*x(il:iu);
      Ax=Ax+A(:,il:iu)*x(il:iu);
  end
% Update s
  cc=x-(1/beta)*z;
  s=max(0,cc);
% Update multipliers
  y=y-beta*(Ax-b);
  z=z-beta*(x-s);
%
  iter=iter+1;
  if (norm(A*x-b)+norm(x-s))/(1+norm(x))<toler, break, end;
end;
toc
x=s;
%    