%======================================================
%  Matlab demonstration of solving quadratic programs
%  by the Randomly Assembled Cyclic Multi-Block ADMM
% (beta = 1):
%      minimize    0.5x'*Q*x+c'*x
%      subject to    A x = b (y dimension m)
%
%  Input 
%      Q: Sparse symmetric objective matrix.
%      A: Sparse constraint left-hand matrix
%      b: constraint right-hand column vector
%      c: objective column vector
%      nb: number of blocks (such that number of 
%          variables divided by nb is an integer)
%      maxiter: the number of iterations
%
%  Output: decision vars x and multipliers y
%
%  Algorithm: Randomly sample variables without replacement
%  to form each block in each cycle, relevant concept in Sect. 
%  14.6 of L&Y, Linear and nonlinear programming, 5th edition
%======================================================%
%
function [x,y,iter]=RACADMMeqp(Q,A,b,c,nb,maxiter)
if exist('maxiter') ~= 1 
   maxiter=500; 
end
tic
toler=1.e-3;
[m,n]=size(A);
beta=1;
x=randn(n,1);
y=zeros(m,1);
nv=floor(n/nb);
% greate two vectos: the gradient of objective and constraint LHS
Qxc=Q*x+c;
Axb=A*x-b;
%
iter=0;
for k=1:maxiter,
  % randomly reorder all variables
  or=randperm(n);
% Update x following a the order block by block
  for ii=1:nb,
      il=(ii-1)*nv+1;
      iu=il+nv-1;
      % find the nv indexes of the (ii)th block according to or
      p=or(il:iu);
      % substract them from the two vectors
      Qxc=Qxc-Q(:,p)*x(p);
      Axb=Axb-A(:,p)*x(p);
      % construct sub-A matrix according to p
      Ap=A(:,p);
      % Construct the gradient vector of x(p)
      cc=Qxc(p)-Ap'*(y-beta*Axb);
      % update x(p)
      x(p)=-(Q(p,p)+beta*(Ap'*Ap))\cc;
      % add them back to the two vectors
      Qxc=Qxc+Q(:,p)*x(p);
      Axb=Axb+A(:,p)*x(p);
      % go to the next block
  end
% Update the multipliers
  y=y-beta*Axb;
%
  iter=iter+1;
  % Check stopping criteria
  if norm(Axb)<toler, break, end;
end;
toc
%     