%======================================================
%  Matlab demonstration of solving quadratic programs
%  by the Randomly Permuted Cyclic Multi-Block ADMM
% (beta = 1):
%      minimize    0.5x'*Q*x+c'*x
%      subject to    A x = b (y dimension m)
%
%  Input 
%      Q: Sparse symmetric objective matrix.
%      A: Sparse constraint left-hand matrix
%      b: constraint right-hand column vector
%      c: objective column vector
%      nb: number of blocks (such that number of 
%          variables divided by nb is an integer)
%      maxiter: the number of iterations
%
%  Output: decision vars x and multipliers y
%
%  Algorithm details can be found in Sect. 14.6
%  L&Y, Linear and nonlinear programming, 5th edition
%======================================================%
function [x,y,iter]=RPCADMMeqp(Q,A,b,c,nb,maxiter)
if exist('maxiter') ~= 1 
   maxiter=500; 
end
tic
toler=1.e-4;
[m,n]=size(A);
beta=1;
x=randn(n,1);
y=randn(m,1);
nv=floor(n/nb);
ATA=zeros(nv,n);
% store block-wise inverse
for ii=1:nb,
    il=(ii-1)*nv+1;
    iu=il+nv-1;
    AB=A(:,il:iu);
    ATA(:,il:iu)=inv(Q(il:iu,il:iu)+beta*(AB'*AB));
end;
clear AB;
% greate two vectos: the gradient of objective and constraint LHS
Qxc=Q*x+c;
Ax=A*x;
%
iter=0;
for k=1:maxiter,
  or=randperm(nb);
  %or=[1 2 3];
% Update x following a random order
  for ii=1:nb,
      il=(or(ii)-1)*nv+1;
      iu=il+nv-1;
      Qxc=Qxc-Q(:,il:iu)*x(il:iu);
      Ax=Ax-A(:,il:iu)*x(il:iu);
      cc=Qxc(il:iu)-A(:,il:iu)'*(y+beta*(b-Ax));
      x(il:iu)=-ATA(:,il:iu)*cc;
      Qxc=Qxc+Q(:,il:iu)*x(il:iu);
      Ax=Ax+A(:,il:iu)*x(il:iu);
  end
% Update multipliers
  y=y-beta*(Ax-b);
%
  iter=iter+1;
  % Check stopping criteria
  if norm(A*x-b)<toler, break, end;
end;
toc
%      