% This is a matlab implementation of the homogeneous
% algorithm for solving the monotone complementarity 
% problem, but it may also solve non-monotone problem:
% 
%      min   s'*x
%      s.t.  s-f(x)=0, (s,x)\ge 0.
%
%  Input 
%      n : the number of variabls
%  toler*: relative stopping tolerance: the objective value close to 
%          the optimal one in the range of tolerance. 
%          Default value: 1.e-6.
%  alpha*: step size: 0 < alpha < 1. Default value: .9.
%  gamma*: weight parameter: 0< gamma <=1. Default value: 1/sqrt(n).
%
%  Matlab functions:
%  mcpfun(x): return the monotone function values at x,
%  mcpjacobian(x): return the jacobian matrix of the function at x.
%     
%  Output
%     x>=0  : 
%     s = mcpfun(x),
%     x^Ts =< toler.
%     OR
% infeasibility certificate for sustem {s-f(x)=0,(s,x)\ge 0}
% 
% Problem/Algorithm details can be found in Sect. 15.7
% of L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
% Set parameters
%
 if exist('toler') ~= 1 
   toler=1.e-6; 
 end
 if exist('gamma') ~= 1 
   gamma=.5;    
 end
 if exist('alpha') ~= 1 
   alpha=0.7;    
 end
%
% Set initial points
%
 n = n+1;
 ee= ones(n,1);
 x = ee;
 s = ee;
 mu = 1;
 iter =0;
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   %
   % Get monotone function values and jacobians
   %
   xx = x(1:n-1)/x(n);
   f = mcpfun(xx);
   G = mcpjacobian(xx);
   %
   % Form the homoginized residuals and jacobians
   %
   r  = [s(1:n-1)-x(n)*f;s(n)+x(1:n-1)'*f];
   MM = [G f-G*xx;-f'-xx'*G xx'*G*xx];
   %
   % Solving one Newton step with the augmented system
   %
   XX=sparse(1:n,1:n,x);
   SS=sparse(1:n,1:n,s);
   %
   % Check dual feasibility residual
   %
   % Linear system solve
   rr=-x.*s+(gamma*mu)*ee+(1-gamma)*(XX*r);
   dx=[XX*MM+SS]\rr;
   %
   % Construct primal and dual steps
   ds=MM*dx-(1-gamma)*r;
%
%  choose step-size
%
   nora = min([dx./x;ds./s]);
   nora = abs(alpha/nora);
%
% Update iterates
%
   x = x + nora*dx;
   s = s + nora*ds;
%   xx = x(1:n-1)/x(n);
%   f =mcpfun(xx,M,c);
%   s = [x(n)*f;-x(1:n-1)'*f]+(1-nora*(1-gamma))*r;
%
% Recompute duality gap
%
   mu = x'*s/n;
 end;
%
% Output solution or infeasibility certificate
%
 iter
 n = n-1;
 tau=x(n+1);
 kappa=s(n+1);
 if kappa < tau,
   x=x(1:n)/tau;
   s=mcpfun(x);
   s=max(0,s);
   disp('Find a complementarity solution');
 else
   x=x(1:n);
   s=max(0,tau*mcpfun(x/tau));
   disp('The problem is infeasible or unattainable');
 end;
 return
%