% Matlab implementation of the second-order interior-point
% primal-dual homotopy or path-following method for solving
% (generalized) linearly constrained (convex) program
%
%      min   f(x,z)
%      s.t.  Ax = b (y), x >= 0, z free.
%
%  Input 
%      A: Sparse constraint matrix.
%      b: right-hand-side vector
%      c: parameter of objective function
%      x0: interior (positive) primal feasile solution with dimension n
%      y0: interior Lagrange multipliers, i.e, nabla_x f(x0,z0)-A'*y0>0
%      z0: initial primal solution of z with dimension p (z0=[] if none)
%      toler: relative stopping tolerance: the objective value close to 
%             the optimal one in the range of tolerance. 
%             Default value: 1.e-6.
%      alpha: step size: alpha > 0. Default value: 1.
%      gamma: reduction rate: 0< gamma <=1. Default value: .7.
%
%  Matlab functions:
%      objgradien(x,z,c): return the gradient vector of the objec. function at 
%      x and z
%      objhessian(x,z,c): return the hessian matrix of the objec. function at 
%      x and z.
%     
%  Output
%     x>=0  : primal nonnegative solution: Ax = b,
%     y,s>=0: Lagrange (dual) solution: s = nabla f(x) - A^Ty,
%             x^Ts/n =< toler.
%     z     : primal free variable solution
% 
% Algorithm details can be found in Sect. 15.5-6
% of L&Y, Linear and nonlinear programming, 5th edition
%======================================================% 
function [x,y,s,z,iter] = IPMpdglccp(A,b,c,x0,y0,z0,toler,alpha,gamma);
% Set parameters
%
 [p,m] = size(z0);
 [m,n] = size(A);
 if exist('toler') ~= 1 
   toler=1.e-6; 
 end
 if exist('alpha') ~= 1 
   alpha=0.3;    
 end
 if exist('gamma') ~= 1 
   gamma=.6;    
 end
%
% Input initial solutions and set slacks
%
 x=x0;
 z=z0;
 y=y0;
 g=objgradien(x,z,c);
 s=g(1:n)-A'*y;
 mu = x'*s/n;
 iter =0; 
%
% Start the loop
%
 while mu >= toler,
   iter = iter + 1;
   %
   mu = gamma*mu;
   ee = ones(n,1);
% Form the right-hand-vector of the KKT system
   qqq = [mu*ee-s.*x;-g(n+1:n+p);-(A*x-b)];
% Form the scaled Hessian/Jacobian matrix of the KKT system
   MM = objhessian(x,z,c);
   XX=sparse(1:n,1:n,x);
   SS=sparse(1:n,1:n,s);
   MM(1:n,1:n)=XX*MM(1:n,1:n)+SS;
% Solve the system of KKT linear equartions
   dx=[MM [-XX*A';sparse(p,m)];A sparse(m,p+m)]\qqq;
% Construct primal and dual directions
   dy=dx(n+p+1:n+p+m);
   dz=dx(n+1:n+p);
   dx=dx(1:n);
   beta = 1; %Newton step-size
   ss = 0;
% Adjust step-size to find an interior solution pair
   while ss <= 0,
    xx = x + beta*dx;
    yy = y + beta*dy;
    zz = z + beta*dz;
    g=objgradien(xx,zz,c);
    s = g(1:n) - A'*yy;
    beta = alpha*beta; % reduce step-size
    ss = min([xx;s]);
   end
   x=xx;
   y=yy;
   z=zz;
   mu = x'*s/n; % reduce mu
 end
 return
 
     



